﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codecommit/CodeCommitRequest.h>
#include <aws/codecommit/CodeCommit_EXPORTS.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodeCommit {
namespace Model {

/**
 */
class PostCommentReplyRequest : public CodeCommitRequest {
 public:
  AWS_CODECOMMIT_API PostCommentReplyRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PostCommentReply"; }

  AWS_CODECOMMIT_API Aws::String SerializePayload() const override;

  AWS_CODECOMMIT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The system-generated ID of the comment to which you want to reply. To get
   * this ID, use <a>GetCommentsForComparedCommit</a> or
   * <a>GetCommentsForPullRequest</a>.</p>
   */
  inline const Aws::String& GetInReplyTo() const { return m_inReplyTo; }
  inline bool InReplyToHasBeenSet() const { return m_inReplyToHasBeenSet; }
  template <typename InReplyToT = Aws::String>
  void SetInReplyTo(InReplyToT&& value) {
    m_inReplyToHasBeenSet = true;
    m_inReplyTo = std::forward<InReplyToT>(value);
  }
  template <typename InReplyToT = Aws::String>
  PostCommentReplyRequest& WithInReplyTo(InReplyToT&& value) {
    SetInReplyTo(std::forward<InReplyToT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, client-generated idempotency token that, when provided in a
   * request, ensures the request cannot be repeated with a changed parameter. If a
   * request is received with the same parameters and a token is included, the
   * request returns information about the initial request that used that token.</p>
   */
  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  PostCommentReplyRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The contents of your reply to a comment.</p>
   */
  inline const Aws::String& GetContent() const { return m_content; }
  inline bool ContentHasBeenSet() const { return m_contentHasBeenSet; }
  template <typename ContentT = Aws::String>
  void SetContent(ContentT&& value) {
    m_contentHasBeenSet = true;
    m_content = std::forward<ContentT>(value);
  }
  template <typename ContentT = Aws::String>
  PostCommentReplyRequest& WithContent(ContentT&& value) {
    SetContent(std::forward<ContentT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_inReplyTo;
  bool m_inReplyToHasBeenSet = false;

  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientRequestTokenHasBeenSet = true;

  Aws::String m_content;
  bool m_contentHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeCommit
}  // namespace Aws
