// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dk_dv.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dk_dv__A0__F82

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_96_0_F_T_0___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_96_0_F_T_0___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_96_0_F_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xc3b112adu, 0x759697a2u, 0, 6 }, // c3b112ad759697a2 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_96_0_F_T_0___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xa1e42dbdu, 0xadfc637du, 0, 63 }, // a1e42dbdadfc637d = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_96_0_F_T_0___gfx950__P__16_16__CO__wave4_warp2_stg1--Arch_gfx950
    { 0x34b68a4au, 0x92ace2afu, 23, 6 }, // 34b68a4a92ace2af = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_96_0_F_T_0___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x3536a7a3u, 0x4cc1ef34u, 23, 46 }, // 3536a7a34cc1ef34 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_96_0_F_T_0___gfx950__P__32_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x10c14a16u, 0x0cd6d2c5u, 23, 63 }, // 10c14a160cd6d2c5 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_96_0_F_T_0___gfx950__P__32_16__CO__wave4_warp2_stg1--Arch_gfx950
    { 0x15faa177u, 0x04e04cbfu, 80, 6 }, // 15faa17704e04cbf = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_96_0_F_T_0___gfx950__P__32_32__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x1670d9e0u, 0x850761f6u, 103, 29 }, // 1670d9e0850761f6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_96_0_F_T_0___gfx950__P__64_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x0d7d7e6fu, 0x450a4643u, 103, 6 }, // 0d7d7e6f450a4643 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_96_0_F_T_0___gfx950__P__64_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x7057e4e5u, 0x962d075au, 103, 126 }, // 7057e4e5962d075a = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_96_0_F_T_0___gfx950__P__64_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x732aaf99u, 0x673a6cdfu, 103, 46 }, // 732aaf99673a6cdf = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_96_0_F_T_0___gfx950__P__64_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x70a345deu, 0x3216c6ecu, 103, 86 }, // 70a345de3216c6ec = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_96_0_F_T_0___gfx950__P__64_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0x7754b41fu, 0x4471e7e9u, 160, 29 }, // 7754b41f4471e7e9 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_96_0_F_T_0___gfx950__P__64_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x00848fc4u, 0x7fb7a878u, 166, 29 }, // 00848fc47fb7a878 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_96_0_F_T_0___gfx950__P__64_64__CO__wave1_warp2_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dk_dv_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dk_dv_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx950_mod0
{{ 3, 2, 3, 2, 2, 3, 1, 2, 1, 0},
 {10,10,10, 8, 2, 2, 3, 3, 3, 2},
 { 8,10,10, 2, 3, 2, 2, 2, 3, 2},
 { 7, 7, 7, 7, 3, 2, 2, 3, 3, 2},
 { 6, 7, 7, 7, 2, 4, 3, 4, 4, 4},
 { 6, 6, 6, 7, 2, 3, 5, 4, 4, 4},
 { 7, 6, 6, 7, 2, 4,12, 5,12,12},
 { 6, 6, 7, 6, 2, 3,12,12,12,12},
 { 6, 6, 7, 6, 3, 7,12,12,11,12},
 { 6, 6, 7, 7,12, 7,12,12, 9,12}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDkDvContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDkDvContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dk_dv__A0__F82 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 0;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dk_dv__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 0;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 96
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = True
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dk_dv WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 96 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

