---- SETUP

-- Table: products
CREATE TABLE products (
    product_id INT,
    product_name STRING,
    product_category STRING,
    product_price DOUBLE,
    product_cost DECIMAL(10, 2),
    supplier_id INT,
    supplier_name STRING,
    supplier_contact STRING,
    supplier_address STRING,
    product_quantity INT,
    product_color STRING,
    product_size STRING,
    product_weight DOUBLE,
    product_description STRING,
    product_rating DOUBLE,
    product_reviews INT,
    product_launch_date TIMESTAMP,
    product_last_update TIMESTAMP,
    product_is_active BOOLEAN,
    product_discount DOUBLE,
    product_vat DOUBLE
);

-- Table: customers
CREATE TABLE customers (
    customer_id INT,
    customer_name STRING,
    customer_email STRING,
    customer_phone STRING,
    customer_address_1 STRING,
    customer_address_2 STRING,
    customer_city STRING,
    customer_state STRING,
    customer_zip STRING,
    customer_country STRING,
    customer_type STRING,
    customer_membership_date TIMESTAMP,
    customer_total_spent DOUBLE,
    customer_last_order_id INT,
    customer_last_order_date TIMESTAMP,
    customer_last_payment_date TIMESTAMP,
    customer_active_status BOOLEAN,
    customer_opt_in_newsletter BOOLEAN,
    customer_discount_percentage DOUBLE,
    customer_vat_number STRING,
    customer_billing_contact STRING
);

-- Table: orders
CREATE TABLE orders (
    order_id INT,
    customer_id INT,
    order_date TIMESTAMP,
    order_status STRING,
    order_total DOUBLE,
    order_tax DOUBLE,
    order_discount DOUBLE,
    order_shipping DOUBLE,
    order_shipping_address_1 STRING,
    order_shipping_address_2 STRING,
    order_shipping_city STRING,
    order_shipping_state STRING,
    order_shipping_zip STRING,
    order_shipping_country STRING,
    order_billing_identifier STRING,
    order_payment_method STRING,
    order_payment_status STRING,
    order_payment_date TIMESTAMP,
    order_cancelation_date TIMESTAMP,
    order_return_request_date TIMESTAMP,
    order_completion_date TIMESTAMP,
    order_source STRING,
    order_device STRING,
    order_channel STRING
);

-- Table: order_items
CREATE TABLE order_items (
    order_item_id INT,
    order_id INT,
    product_id INT,
    item_quantity INT,
    item_price_per_unit DOUBLE,
    item_discount DOUBLE,
    item_tax DOUBLE,
    item_total DOUBLE,
    item_status STRING,
    item_offer_code STRING,
    item_shipping_date TIMESTAMP,
    item_delivery_date TIMESTAMP,
    item_return_date TIMESTAMP
);

-- Table: employees
CREATE TABLE employees (
    employee_id INT,
    employee_name STRING,
    employee_email STRING,
    employee_phone STRING,
    employee_position STRING,
    employee_department STRING,
    employee_hire_date TIMESTAMP,
    employee_termination_date TIMESTAMP,
    employee_salary DOUBLE,
    employee_commission DOUBLE,
    employee_manager_id INT,
    employee_address_1 STRING,
    employee_address_2 STRING,
    employee_city STRING,
    employee_state STRING,
    employee_zip STRING,
    employee_country STRING,
    employee_birth_date TIMESTAMP,
    employee_gender STRING,
    employee_marital_status STRING,
    employee_dependents INT
);

-- Table: suppliers
CREATE TABLE suppliers (
    supplier_id INT,
    supplier_name STRING,
    supplier_contact STRING,
    supplier_contact_role STRING,
    supplier_contact_email STRING,
    supplier_contact_phone_1 STRING,
    supplier_contact_phone_2 STRING,
    supplier_address_1 STRING,
    supplier_address_2 STRING,
    supplier_address_3 STRING,
    supplier_city STRING,
    supplier_state STRING,
    supplier_zip STRING,
    supplier_country STRING,
    supplier_region STRING,
    supplier_main_products STRING,
    supplier_website STRING,
    supplier_pricing_agreement_date TIMESTAMP,
    supplier_last_order_date TIMESTAMP,
    supplier_last_contact_date TIMESTAMP,
    supplier_rating DOUBLE,
    supplier_active BOOLEAN
);

-- INSERT example data (for each table, a few typical rows)
INSERT INTO products VALUES (1, 'Super Widget', 'Electronics', 155.99, 99.99, 1, 'Acme Inc', 'John D.', '123 Main St', 25, 'Blue', 'Large', 1.2, 'High quality widget', 4.5, 120, TIMESTAMP '2022-01-01', TIMESTAMP '2023-01-01', true, 0.0, 0.21);
INSERT INTO products VALUES (2, 'Deluxe Gadget', 'Electronics', 89.99, 60.00, 2, 'Widgets Co', 'Mary K.', '456 Elm St', 100, 'Red', 'Small', 0.5, 'Portable gadget', 4.2, 200, TIMESTAMP '2022-02-15', TIMESTAMP '2023-02-15', true, 0.1, 0.19);
INSERT INTO products VALUES (3, 'Classic Toy', 'Toys', 19.95, 12.00, 3, 'Toy Supply', 'Ann T.', '789 Oak St', 50, 'Multicolor', 'Medium', 0.8, 'Classic wooden toy', 4.7, 80, TIMESTAMP '2021-11-01', TIMESTAMP '2023-01-10', true, 0.05, 0.15);

INSERT INTO customers VALUES (1, 'Alice Johnson', 'alice@example.com', '555-1000', '101 Maple Ave', NULL, 'Springfield', 'IL', '12345', 'USA', 'Retail', TIMESTAMP '2020-05-01', 899.99, 1001, TIMESTAMP '2023-01-20', TIMESTAMP '2023-01-21', true, true, 5.0, 'US123456', 'Alice Johnson');
INSERT INTO customers VALUES (2, 'Bob Smith', 'bob@example.com', '555-1002', '202 Oak St', 'Apt 3', 'Oakville', 'CA', '67890', 'USA', 'Wholesale', TIMESTAMP '2021-03-15', 1200.50, 1002, TIMESTAMP '2023-02-10', TIMESTAMP '2023-02-11', true, false, 10.0, 'US923456', 'Bob Smith');
INSERT INTO customers VALUES (3, 'Cathy Lee', 'cathy@example.com', '555-1003', '303 Pine Ln', NULL, 'Pineville', 'TX', '54321', 'USA', 'Retail', TIMESTAMP '2022-07-01', 400.75, 1003, TIMESTAMP '2023-03-05', TIMESTAMP '2023-03-06', true, true, 0.0, 'US563456', 'Cathy Lee');

INSERT INTO orders VALUES (1001, 1, TIMESTAMP '2023-01-20', 'Completed', 155.99, 15.89, 0.0, 10.00, '101 Maple Ave', NULL, 'Springfield', 'IL', '12345', 'USA', 'ORD1001', 'Credit Card', 'Paid', TIMESTAMP '2023-01-21', NULL, NULL, TIMESTAMP '2023-01-25', 'Web', 'Desktop', 'Online');
INSERT INTO orders VALUES (1002, 2, TIMESTAMP '2023-02-10', 'Completed', 179.89, 18.99, 10.0, 8.99, '202 Oak St', 'Apt 3', 'Oakville', 'CA', '67890', 'USA', 'ORD1002', 'Credit Card', 'Paid', TIMESTAMP '2023-02-11', NULL, NULL, TIMESTAMP '2023-02-15', 'Mobile', 'iOS', 'Online');
INSERT INTO orders VALUES (1003, 3, TIMESTAMP '2023-03-05', 'Processing', 19.95, 3.99, 1.0, 5.99, '303 Pine Ln', NULL, 'Pineville', 'TX', '54321', 'USA', 'ORD1003', 'PayPal', 'Pending', NULL, NULL, NULL, NULL, 'App', 'Android', 'Mobile App');

INSERT INTO order_items VALUES (101, 1001, 1, 1, 155.99, 0.0, 15.89, 171.88, 'Shipped', 'SPRING2023', TIMESTAMP '2023-01-20', NULL, NULL);
INSERT INTO order_items VALUES (102, 1002, 2, 2, 89.99, 8.99, 8.99, 170.97, 'Shipped', 'SUMMER2023', TIMESTAMP '2023-02-10', NULL, NULL);
INSERT INTO order_items VALUES (103, 1003, 3, 1, 19.95, 1.0, 3.99, 22.94, 'Processing', 'FALL2023', NULL, NULL, NULL);

INSERT INTO employees VALUES (1, 'Dan Miller', 'dan@example.com', '555-2001', 'Manager', 'Sales', TIMESTAMP '2018-01-01', NULL, 75000.0, 2500.0, NULL, '404 Birch St', NULL, 'Birchville', 'NY', '00123', 'USA', TIMESTAMP '1980-07-10', 'Male', 'Married', 2);
INSERT INTO employees VALUES (2, 'Eva Perez', 'eva@example.com', '555-2002', 'Salesperson', 'Sales', TIMESTAMP '2019-03-01', NULL, 50000.0, 1500.0, 1, '505 Cedar Dr', 'Unit 2', 'Cedarville', 'NJ', '00234', 'USA', TIMESTAMP '1992-09-25', 'Female', 'Single', 0);
INSERT INTO employees VALUES (3, 'Frank Wong', 'frank@example.com', '555-2003', 'Warehouse', 'Operations', TIMESTAMP '2020-06-15', NULL, 40000.0, 0.0, 2, '606 Walnut Ln', NULL, 'Walnut', 'CA', '00345', 'USA', TIMESTAMP '1988-02-14', 'Male', 'Married', 1);

INSERT INTO suppliers VALUES (1, 'Acme Inc', 'John D.', 'Sales Manager', 'john@acme.com', '555-3001', '555-3002', '123 Main St', NULL, NULL, 'Springfield', 'IL', '12345', 'USA', 'Midwest', 'Widgets, Gadgets', 'https://acme.com', TIMESTAMP '2020-01-01', NULL, TIMESTAMP '2023-01-10', 4.8, true);
INSERT INTO suppliers VALUES (2, 'Widgets Co', 'Mary K.', 'Customer Success', 'mary@widgets.com', '555-4001', NULL, '456 Elm St', NULL, NULL, 'Oakville', 'CA', '67890', 'USA', 'West', 'Gadgets, Toys', 'https://widgetsco.com', TIMESTAMP '2021-02-01', NULL, TIMESTAMP '2023-02-15', 4.6, true);
INSERT INTO suppliers VALUES (3, 'Toy Supply', 'Ann T.', 'Director', 'ann@toysupply.com', '555-5001', NULL, '789 Oak St', NULL, NULL, 'Pineville', 'TX', '54321', 'USA', 'South', 'Toys, Games', 'https://toysupply.com', TIMESTAMP '2019-03-01', NULL, TIMESTAMP '2023-03-01', 4.4, true);

---- Scripts

-- Script 1
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_inventory_threshold INT DEFAULT 20;
    DECLARE v_low_stock_products INT DEFAULT 0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Stock error'); END;
    -- Nested compound statements, labels, mixed control flow
    main_block: BEGIN
        DECLARE v_product_id INT;
        DECLARE v_product_name STRING;
        DECLARE v_supplier_id INT;
        DECLARE v_action_taken STRING;
        FOR product AS SELECT p.product_id, p.product_name, p.supplier_id, p.product_quantity FROM products p JOIN suppliers s ON p.supplier_id = s.supplier_id LIMIT 5 DO
            SET v_product_id = product.product_id;
            SET v_product_name = product.product_name;
            SET v_supplier_id = product.supplier_id;
            L1: BEGIN
                CASE
                    WHEN product.product_quantity < v_inventory_threshold THEN
                        SET v_action_taken = 'Restock needed';
                        SET v_low_stock_products = v_low_stock_products + 1;
                        INSERT INTO orders (order_id, customer_id, order_date, order_status, order_total, order_tax, order_discount, order_shipping)
                        VALUES (v_product_id * 1000, (SELECT MIN(customer_id) FROM customers), CURRENT_TIMESTAMP, 'Restock', (SELECT product_cost FROM products WHERE product_id = v_product_id) * 10, 0.0, 0.0, 0.0);
                        INSERT INTO order_items (order_item_id, order_id, product_id, item_quantity, item_price_per_unit, item_discount, item_tax, item_total)
                        VALUES (v_product_id * 100 + 1, v_product_id * 1000, v_product_id, 10, (SELECT product_cost FROM products WHERE product_id = v_product_id), 0.0, 0.0, (SELECT product_cost FROM products WHERE product_id = v_product_id) * 10);
                    ELSE
                        SET v_action_taken = 'Sufficient stock';
                END CASE;
                BEGIN
                    DECLARE v_supplier_active BOOLEAN;
                    SET v_supplier_active = (SELECT supplier_active FROM suppliers WHERE supplier_id = v_supplier_id);
                    IF NOT v_supplier_active THEN
                        SET v_action_taken = CONCAT(v_action_taken, ' (Supplier inactive)');
                        BEGIN
                            DECLARE v_temp_id INT;
                            SET v_temp_id = (SELECT COALESCE(MAX(supplier_id), 0) + 1 FROM suppliers);
                            INSERT INTO suppliers VALUES (v_temp_id, 'Temp Supplier', 'Temp Contact', 'Temp Role', 'temp@temp.com', '555-0000', NULL, 'Temp Address', NULL, NULL, 'Temp City', 'TX', '00000', 'USA', 'South', 'Temporary', 'http://temp.com', CURRENT_TIMESTAMP, NULL, CURRENT_TIMESTAMP, 3.0, true);
                        END;
                    END IF;
                END;
            END L1;
        END FOR;
    END main_block;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, CASE, IF, DML, nested-scopes, exception-handler, mixed-logic
-- EXPECTED: Random restock orders, supplier inserts, status logic
-- EXECUTES: FOR, CASE, IF, nested BEGIN...END

-- Script 2
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_city_count INT;
    DECLARE v_state_count INT;
    DECLARE v_city_shortage_found BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Geo error'); END;
    -- Deep, wide, and random nesting with labels
    city_loop: FOR city AS SELECT DISTINCT customer_city FROM customers LIMIT 3 DO
        SET v_city_count = (SELECT COUNT(*) FROM customers WHERE customer_city = city.customer_city);
        state_scope: BEGIN
            DECLARE v_state STRING;
            SET v_state = (SELECT MAX(customer_state) FROM customers WHERE customer_city = city.customer_city);
            IF v_state IS NOT NULL THEN
                SET v_state_count = (SELECT COUNT(*) FROM customers WHERE customer_state = v_state);
                L1: BEGIN
                    DECLARE v_region_ok BOOLEAN DEFAULT true;
                    FOR region AS SELECT DISTINCT supplier_region FROM suppliers LIMIT 1 DO
                        IF region.supplier_region IS NULL THEN
                            SET v_region_ok = false;
                        END IF;
                    END FOR;
                    IF NOT v_region_ok THEN
                        SET v_city_shortage_found = true;
                    END IF;
                END L1;
            END IF;
        END state_scope;
        BEGIN
            DECLARE v_employee_id INT;
            SET v_employee_id = (SELECT MIN(employee_id) FROM employees);
            IF v_city_count < 2 AND NOT v_city_shortage_found THEN
                BEGIN
                    DECLARE v_new_customer_id INT;
                    SET v_new_customer_id = (SELECT COALESCE(MAX(customer_id), 0) + 1 FROM customers);
                    INSERT INTO customers VALUES (v_new_customer_id, 'New Customer', 'new@customer.com', '555-1111', '555 Place', city.customer_city, (SELECT DISTINCT customer_state FROM customers WHERE customer_city = city.customer_city LIMIT 1), '00000', 'USA', 'Retail', CURRENT_TIMESTAMP, 0.0, NULL, NULL, NULL, true, true, 0.0, NULL, NULL);
                END;
            END IF;
            WHILE v_employee_id < 3 DO
                SET v_employee_id = v_employee_id + 1;
                BEGIN
                    DECLARE v_customer_address STRING;
                    SET v_customer_address = (SELECT customer_address_1 FROM customers WHERE customer_city = city.customer_city LIMIT 1);
                    IF v_customer_address IS NULL THEN
                        SET v_customer_address = 'Unknown';
                    END IF;
                END;
            END WHILE;
        END;
    END FOR city_loop;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, WHILE, DML, deep-nesting, wide-scopes, exception-handler, geo-logic
-- EXPECTED: Random customer inserts, geo checks, possible city shortage
-- EXECUTES: FOR, IF, WHILE, nested BEGIN...END, exception handler

-- Script 3
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_total_orders INT;
    DECLARE v_customer_id INT;
    DECLARE v_order_id INT;
    DECLARE v_last_order_date TIMESTAMP;
    DECLARE v_product_count INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Orders processing error'); END;
    -- Wide and deep, with mixed control structures
    customer_block: BEGIN
        DECLARE v_customer_threshold INT DEFAULT 2;
        FOR cust AS SELECT customer_id, customer_name FROM customers LIMIT 5 DO
            SET v_customer_id = cust.customer_id;
            SET v_total_orders = (SELECT COUNT(*) FROM orders WHERE customer_id = cust.customer_id);
            SET v_last_order_date = (SELECT MAX(order_date) FROM orders WHERE customer_id = cust.customer_id);
            IF v_total_orders < v_customer_threshold THEN
                SET v_order_id = (SELECT COALESCE(MAX(order_id), 0) + 1 FROM orders);
                INSERT INTO orders (order_id, customer_id, order_date, order_status, order_total, order_tax, order_discount, order_shipping)
                VALUES (v_order_id, v_customer_id, CURRENT_TIMESTAMP, 'New', 10.0, 1.0, 0.0, 2.0);
                L1: BEGIN
                    DECLARE v_item_id INT;
                    SET v_product_count = (SELECT COUNT(*) FROM products);
                    IF v_product_count > 0 THEN
                        FOR prod AS SELECT product_id FROM products LIMIT 3 DO
                            SET v_item_id = (SELECT COALESCE(MAX(order_item_id), 0) + 1 FROM order_items);
                            INSERT INTO order_items (order_item_id, order_id, product_id, item_quantity, item_price_per_unit, item_discount, item_tax, item_total)
                            VALUES (v_item_id, v_order_id, prod.product_id, 1, (SELECT product_price FROM products WHERE product_id = prod.product_id), 0.0, 0.0, (SELECT product_price FROM products WHERE product_id = prod.product_id));
                        END FOR;
                    END IF;
                END L1;
            ELSE
                L2: BEGIN
                    DECLARE v_days_since_last_order INT;
                    SET v_days_since_last_order = DATEDIFF(DAY, v_last_order_date, CURRENT_TIMESTAMP);
                    IF v_days_since_last_order > 30 THEN
                        L3: BEGIN
                            DECLARE v_random_product_id INT;
                            SET v_random_product_id = (SELECT product_id FROM products LIMIT 1);
                            INSERT INTO order_items (order_item_id, order_id, product_id, item_quantity, item_price_per_unit, item_discount, item_tax, item_total)
                            VALUES ((SELECT COALESCE(MAX(order_item_id), 0) + 1 FROM order_items), (SELECT MIN(order_id) FROM orders WHERE customer_id = v_customer_id), v_random_product_id, 1, (SELECT product_price FROM products WHERE product_id = v_random_product_id), 0.0, 0.0, (SELECT product_price FROM products WHERE product_id = v_random_product_id));
                        END L3;
                        SET v_last_order_date = CURRENT_TIMESTAMP;
                    ELSE
                        BEGIN
                            DECLARE v_orders_to_delete INT;
                            SET v_orders_to_delete = (SELECT COUNT(*) FROM orders WHERE order_status = 'Processing' AND customer_id = v_customer_id);
                            WHILE v_orders_to_delete > 1 DO
                                DELETE FROM orders WHERE order_id IN (SELECT order_id FROM orders WHERE order_status = 'Processing' AND customer_id = v_customer_id LIMIT 1);
                                SET v_orders_to_delete = v_orders_to_delete - 1;
                            END WHILE;
                        END;
                    END IF;
                END L2;
            END IF;
        END FOR;
    END customer_block;
    BEGIN
        DECLARE v_max_order_total DOUBLE;
        SET v_max_order_total = (SELECT MAX(order_total) FROM orders);
        IF v_max_order_total > 200 THEN
            L4: BEGIN
                DECLARE v_large_orders INT;
                SET v_large_orders = (SELECT COUNT(*) FROM orders WHERE order_total > 200);
                IF v_large_orders > 1 THEN
                    REPEAT
                        UPDATE orders SET order_shipping = order_shipping * 0.9 WHERE order_status = 'Completed' AND order_total > 200;
                        SET v_large_orders = (SELECT COUNT(*) FROM orders WHERE order_total > 200);
                    UNTIL v_large_orders < 2 END REPEAT;
                END IF;
            END L4;
        END IF;
    END;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, WHILE, REPEAT, DML, nested-scopes, exception-handler, mixed-logic
-- EXPECTED: Orders and items created, updated, or deleted; conditional logic on customer activity
-- EXECUTES: FOR, IF, WHILE, REPEAT, nested BEGIN...END, exception handler

-- Script 4
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_first_name STRING;
    DECLARE v_last_name STRING;
    DECLARE v_employee_name STRING;
    DECLARE v_is_manager BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Employee processing error'); END;
    -- Deep, wide, variable nesting, labels
    staff_scope: BEGIN
        DECLARE v_emp_count INT DEFAULT 0;
        DECLARE v_manager_id INT;
        SET v_manager_id = (SELECT MIN(employee_id) FROM employees WHERE employee_position = 'Manager');
        FOR emp AS SELECT employee_id, employee_name, employee_position FROM employees LIMIT 4 DO
            SET v_employee_name = emp.employee_name;
            SET v_emp_count = v_emp_count + 1;
            L1: BEGIN
                DECLARE v_split_result ARRAY<STRING>;
                SET v_split_result = split(v_employee_name, ' ');
                CASE v_split_result[0]
                    WHEN 'Dan' THEN SET v_is_manager = true;
                    ELSE SET v_is_manager = false;
                END CASE;
                IF v_is_manager AND v_emp_count > 1 THEN
                    L2: BEGIN
                        DECLARE v_subordinates INT;
                        SET v_subordinates = (SELECT COUNT(*) FROM employees WHERE employee_manager_id = emp.employee_id);
                        IF v_subordinates > 0 THEN
                            FOR sub_emp AS SELECT employee_id, employee_name FROM employees WHERE employee_manager_id = emp.employee_id LIMIT 2 DO
                                L3: BEGIN
                                    DECLARE v_name_part STRING;
                                    SET v_name_part = (SELECT CONCAT(split(sub_emp.employee_name, ' ')[0], '_', ROUND(RANDOM() * 100)));
                                    INSERT INTO customers (customer_id, customer_name, customer_email, customer_phone, customer_address_1, customer_city, customer_state, customer_zip, customer_country, customer_active_status)
                                    VALUES (sub_emp.employee_id + 9999, v_name_part, CONCAT(v_name_part, '@company.com'), '555-8888', '123 HR St', 'HQ City', 'CA', '00000', 'USA', true);
                                END L3;
                            END FOR;
                        END IF;
                    END L2;
                ELSE
                    REPEAT
                        SET v_emp_count = v_emp_count + 1;
                        L4: BEGIN
                            DECLARE v_todays_date STRING;
                            SET v_todays_date = DATE_FORMAT(CURRENT_TIMESTAMP, 'yyyy-MM-dd');
                            INSERT INTO suppliers (supplier_id, supplier_name, supplier_contact, supplier_contact_role, supplier_contact_email, supplier_contact_phone_1, supplier_address_1, supplier_city, supplier_state, supplier_zip, supplier_country, supplier_region, supplier_main_products, supplier_active)
                            VALUES (emp.employee_id + 10000, CONCAT('Emp_', emp.employee_id), emp.employee_name, 'Employee', CONCAT(emp.employee_name, '@temp.com'), '555-9999', '123 Emp St', 'Emp City', 'CA', '00001', 'USA', 'West', 'Misc', true);
                        END L4;
                    UNTIL v_emp_count > 3 END REPEAT;
                END IF;
            END L1;
        END FOR;
    END staff_scope;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, CASE, REPEAT, DML, nested-scopes, exception-handler, string-manipulation
-- EXPECTED: Customer and supplier inserts based on staff data, string manipulation, manager checks
-- EXECUTES: FOR, IF, CASE, REPEAT, nested BEGIN...END, exception handler

-- Script 5
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_category STRING;
    DECLARE v_total_reviews INT DEFAULT 0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Product review error'); END;
    -- Wide, deep, random nested logic
    prod_category: BEGIN
        DECLARE v_category_count INT;
        SET v_category_count = (SELECT COUNT(DISTINCT product_category) FROM products);
        FOR cat AS SELECT DISTINCT product_category FROM products LIMIT 3 DO
            SET v_category = cat.product_category;
            SET v_total_reviews = (SELECT SUM(product_reviews) FROM products WHERE product_category = cat.product_category);
            IF v_total_reviews > 0 THEN
                L1: BEGIN
                    DECLARE v_average_rating DOUBLE;
                    SET v_average_rating = (SELECT AVG(product_rating) FROM products WHERE product_category = cat.product_category);
                    BEGIN
                        DECLARE v_price_range DOUBLE;
                        SET v_price_range = (SELECT MAX(product_price) - MIN(product_price) FROM products WHERE product_category = cat.product_category);
                        IF v_average_rating > 4.5 AND v_price_range > 50 THEN
                            L2: BEGIN
                                DECLARE v_updated_products INT DEFAULT 0;
                                FOR prod AS SELECT product_id FROM products WHERE product_category = cat.product_category LIMIT 2 DO
                                    UPDATE products SET product_price = product_price * 0.95, product_description = CONCAT('***Premium*** ', product_description) WHERE product_id = prod.product_id;
                                    SET v_updated_products = v_updated_products + 1;
                                END FOR;
                            END L2;
                        ELSE
                            BEGIN
                                DECLARE v_new_product_id INT;
                                SET v_new_product_id = (SELECT COALESCE(MAX(product_id), 0) + 1 FROM products);
                                INSERT INTO products VALUES (v_new_product_id, 'New ' || v_category, v_category, 49.99, 39.99, 2, 'Widgets Co', 'Sales', '123 Widget St', 30, 'Black', 'Medium', 1.5, 'New arrival', 5.0, 0, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP, true, 0.0, 0.19);
                            END;
                        END IF;
                    END;
                END L1;
            ELSE
                L3: BEGIN
                    DECLARE v_fake_reviews INT DEFAULT 0;
                    REPEAT
                        UPDATE products SET product_reviews = product_reviews + 5, product_rating = LEAST(product_rating + 0.1, 5.0) WHERE product_category = cat.product_category AND product_reviews < 10;
                        SET v_fake_reviews = v_fake_reviews + 1;
                    UNTIL v_fake_reviews > 2 END REPEAT;
                END L3;
            END IF;
        END FOR;
    END prod_category;
    BEGIN
        DECLARE v_total_product_types INT;
        SET v_total_product_types = (SELECT COUNT(DISTINCT product_name) FROM products);
        IF v_total_product_types < 10 THEN
            L4: BEGIN
                DECLARE v_missing_color STRING;
                SET v_missing_color = 'Red';
                IF NOT EXISTS (SELECT 1 FROM products WHERE product_color = v_missing_color) THEN
                    INSERT INTO products VALUES ((SELECT COALESCE(MAX(product_id), 0) + 1 FROM products), 'Rare ' || v_missing_color, 'Special', 99.99, 69.99, 1, 'Acme Inc', 'Sales', '123 Rare St', 2, v_missing_color, 'Unique', 2.0, 'Extremely rare', 5.0, 0, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP, true, 0.0, 0.21);
                END IF;
            END L4;
        END IF;
    END;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, nested-scopes, exception-handler, string-manipulation, arithmetic-logic
-- EXPECTED: Random product updates, inserts, and deletes; fake reviews if needed; checks for missing colors
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 6
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_hq_state STRING DEFAULT 'CA';
    DECLARE v_regional_office_count INT DEFAULT 0;
    DECLARE v_wrong_city_found BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Office location error'); END;

    -- Wide, deep, random logic with labels
    main_office: BEGIN
        DECLARE v_total_offices INT;
        SET v_total_offices = (SELECT COUNT(DISTINCT customer_city) FROM customers);
        FOR city AS SELECT DISTINCT customer_city FROM customers LIMIT 2 DO
            SET v_wrong_city_found = true;
            IF city.customer_city LIKE '%ville' THEN
                SET v_wrong_city_found = false;
                L1: BEGIN
                    DECLARE v_region_check BOOLEAN DEFAULT true;
                    FOR supplier_region AS SELECT supplier_region FROM suppliers WHERE supplier_region LIKE '%est%' LIMIT 1 DO
                        IF supplier_region IS NULL THEN
                            SET v_region_check = false;
                        END IF;
                    END FOR;
                    IF NOT v_region_check THEN
                        INSERT INTO suppliers (
                            supplier_id, supplier_name, supplier_contact, supplier_contact_role,
                            supplier_contact_email, supplier_contact_phone_1, supplier_address_1,
                            supplier_city, supplier_state, supplier_zip, supplier_country, supplier_region
                        ) VALUES (
                            (SELECT MAX(supplier_id) + 1 FROM suppliers),
                            'Regional Supplier',
                            'Regional Contact',
                            'Regional Manager',
                            'region@supplier.com',
                            '555-1212',
                            '123 Regional St',
                            city.customer_city,
                            'CA',
                            '00001',
                            'USA',
                            'West'
                        );
                    END IF;
                END L1;
                SET v_regional_office_count = v_regional_office_count + 1;
            END IF;
        END FOR;
    END main_office;
    BEGIN
        DECLARE v_hq_exists BOOLEAN;
        SET v_hq_exists = (SELECT COUNT(*) FROM suppliers WHERE supplier_state = v_hq_state);
        IF NOT v_hq_exists OR v_wrong_city_found THEN
            L2: BEGIN
                DECLARE v_emp_in_city INT;
                SET v_emp_in_city = (SELECT COUNT(*) FROM employees WHERE employee_state = v_hq_state AND employee_city LIKE '%town%');
                IF v_emp_in_city > 0 THEN
                    REPEAT
                        INSERT INTO suppliers (
                            supplier_id, supplier_name, supplier_contact, supplier_contact_role,
                            supplier_contact_email, supplier_contact_phone_1, supplier_address_1,
                            supplier_city, supplier_state, supplier_zip, supplier_country, supplier_region, supplier_active
                        ) VALUES (
                            (SELECT MAX(supplier_id) + 1 FROM suppliers),
                            'Main Office Supplier',
                            'HQ Contact',
                            'HQ Manager',
                            'hq@supplier.com',
                            '555-1111',
                            '123 Main St',
                            (SELECT employee_city FROM employees WHERE employee_state = v_hq_state LIMIT 1),
                            v_hq_state,
                            '00000',
                            'USA',
                            'HQ',
                            true
                        );
                        SET v_hq_exists = (SELECT COUNT(*) FROM suppliers WHERE supplier_state = v_hq_state);
                    UNTIL v_hq_exists = 0 OR v_hq_exists > 2 END REPEAT;
                END IF;
            END L2;
        END IF;
    END;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, geo-logic
-- EXPECTED: Random supplier inserts based on city, region, and state logic
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 7
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_total_orders INT DEFAULT 0;
    DECLARE v_order_counter INT DEFAULT 0;
    DECLARE v_order_status_count INT DEFAULT 0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Order status analysis error'); END;

    -- Wide, deep, random nesting, labels
    BEGIN
        DECLARE v_limit_orders INT DEFAULT 3;
        FOR status AS SELECT DISTINCT order_status FROM orders LIMIT 2 DO
            SET v_order_counter = (SELECT COUNT(*) FROM orders WHERE order_status = status.order_status);
            IF v_order_counter > 0 THEN
                L1: BEGIN
                    DECLARE v_avg_order_total DOUBLE;
                    SET v_avg_order_total = (SELECT AVG(order_total) FROM orders WHERE order_status = status.order_status);
                    IF v_avg_order_total > 100 THEN
                        L2: BEGIN
                            DECLARE v_order_tax_avg DOUBLE;
                            SET v_order_tax_avg = (SELECT AVG(order_tax) FROM orders WHERE order_status = status.order_status);
                            IF v_order_tax_avg > 10 THEN
                                REPEAT
                                    UPDATE orders SET order_tax = order_tax * 0.95 WHERE order_status = status.order_status;
                                    SET v_order_tax_avg = (SELECT AVG(order_tax) FROM orders WHERE order_status = status.order_status);
                                UNTIL v_order_tax_avg <= 10 END REPEAT;
                            END IF;
                            SET v_limit_orders = v_limit_orders + 1;
                        END L2;
                    ELSE
                        BEGIN
                            DECLARE v_new_order_id INT;
                            SET v_new_order_id = (SELECT COALESCE(MAX(order_id), 0) + 1 FROM orders);
                            INSERT INTO orders (
                                order_id, customer_id, order_date, order_status, order_total, order_tax, order_discount, order_shipping
                            ) VALUES (
                                v_new_order_id,
                                (SELECT MAX(customer_id) FROM customers),
                                CURRENT_TIMESTAMP,
                                status.order_status,
                                100.0,
                                10.0,
                                0.0,
                                2.0
                            );
                            SET v_limit_orders = v_limit_orders - 1;
                        END;
                    END IF;
                END L1;
            END IF;
            SET v_order_status_count = v_order_status_count + 1;
        END FOR;
        WHILE v_order_status_count < 3 DO
            BEGIN
                DECLARE v_fake_status STRING;
                SET v_fake_status = CONCAT('Fake_Status_', v_order_status_count);
                INSERT INTO orders (
                    order_id, customer_id, order_date, order_status, order_total, order_tax, order_discount, order_shipping
                ) VALUES (
                    (SELECT COALESCE(MAX(order_id), 0) + 1 FROM orders),
                    (SELECT MIN(customer_id) FROM customers),
                    CURRENT_TIMESTAMP,
                    v_fake_status,
                    10.0,
                    1.0,
                    0.0,
                    2.0
                );
                SET v_order_status_count = v_order_status_count + 1;
            END;
        END WHILE;
    END;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, WHILE, REPEAT, DML, deep-nesting, exception-handler, order-status-logic
-- EXPECTED: Random order status updates, inserts, and status logic
-- EXECUTES: FOR, IF, WHILE, REPEAT, nested BEGIN...END, exception handler

-- Script 8
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_supplier_id INT;
    DECLARE v_supplier_active BOOLEAN;
    DECLARE v_contact_orders INT DEFAULT 0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Supplier contact management error'); END;

    -- Wide, deep, random logic, labels
    supplier_loop: FOR supplier AS SELECT supplier_id, supplier_active, supplier_contact FROM suppliers LIMIT 3 DO
        SET v_supplier_id = supplier.supplier_id;
        SET v_supplier_active = supplier.supplier_active;
        IF v_supplier_active THEN
            BEGIN
                DECLARE v_contact_has_orders BOOLEAN DEFAULT false;
                SET v_contact_orders = (SELECT COUNT(*) FROM products WHERE supplier_id = v_supplier_id);
                IF v_contact_orders > 0 THEN
                    SET v_contact_has_orders = true;
                    L1: BEGIN
                        DECLARE v_new_product_id INT;
                        SET v_new_product_id = (SELECT COALESCE(MAX(product_id), 0) + 1 FROM products);
                        INSERT INTO products VALUES (
                            v_new_product_id,
                            'Supplementary Item',
                            'Accessories',
                            19.99,
                            12.99,
                            v_supplier_id,
                            supplier.supplier_contact,
                            supplier.supplier_contact,
                            'Extra St',
                            10,
                            'Black',
                            'Medium',
                            1.2,
                            'Supplementary product',
                            4.0,
                            0,
                            CURRENT_TIMESTAMP,
                            CURRENT_TIMESTAMP,
                            true,
                            0.0,
                            0.19
                        );
                        BEGIN
                            DECLARE v_days_since_last_order INT;
                            SET v_days_since_last_order = DATEDIFF(DAY, (SELECT MAX(product_launch_date) FROM products WHERE supplier_id = v_supplier_id), CURRENT_TIMESTAMP);
                            IF v_days_since_last_order > 30 THEN
                                L2: BEGIN
                                    DECLARE v_temp_id INT;
                                    REPEAT
                                        SET v_temp_id = (SELECT MIN(supplier_id) FROM suppliers WHERE supplier_active = true AND supplier_id != v_supplier_id);
                                        IF v_temp_id IS NOT NULL THEN
                                            UPDATE products SET supplier_id = v_temp_id WHERE product_id = v_new_product_id;
                                        END IF;
                                    UNTIL v_temp_id IS NULL OR v_days_since_last_order < 20 END REPEAT;
                                END L2;
                            END IF;
                        END;
                    END L1;
                ELSE
                    L3: BEGIN
                        DECLARE v_supplier_emails INT;
                        SET v_supplier_emails = (SELECT COUNT(*) FROM suppliers WHERE supplier_contact_email IS NOT NULL AND supplier_id = v_supplier_id);
                        IF v_supplier_emails = 0 THEN
                            BEGIN
                                DECLARE v_fake_email STRING;
                                SET v_fake_email = CONCAT(supplier.supplier_contact, '@fake.com');
                                INSERT INTO customers (
                                    customer_id, customer_name, customer_email, customer_phone,
                                    customer_address_1, customer_city, customer_state, customer_zip,
                                    customer_country, customer_active_status
                                ) VALUES (
                                    v_supplier_id + 9000,
                                    supplier.supplier_contact,
                                    v_fake_email,
                                    '555-7777',
                                    '123 Fake St',
                                    'Fakeville',
                                    'CA',
                                    '00000',
                                    'USA',
                                    false
                                );
                            END;
                        END IF;
                    END L3;
                END IF;
                BEGIN
                    DECLARE v_orders_with_supplier INT;
                    SET v_orders_with_supplier = (SELECT COUNT(*) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE p.supplier_id = v_supplier_id);
                    IF v_orders_with_supplier > 0 THEN
                        L4: BEGIN
                            DECLARE v_total_value DOUBLE;
                            SET v_total_value = (SELECT SUM(oi.item_total) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE p.supplier_id = v_supplier_id);
                            IF v_total_value > 200 THEN
                                BEGIN
                                    DECLARE v_new_discount DOUBLE;
                                    SET v_new_discount = CASE WHEN v_total_value < 500 THEN 0.05 ELSE 0.10 END;
                                    UPDATE suppliers SET supplier_rating = supplier_rating + 0.1 WHERE supplier_id = v_supplier_id;
                                    UPDATE products SET product_discount = product_discount + v_new_discount WHERE supplier_id = v_supplier_id;
                                END;
                            END IF;
                        END L4;
                    END IF;
                END;
            END;
        ELSE
            L5: BEGIN
                DECLARE v_inactive_supplier_orders INT;
                SET v_inactive_supplier_orders = (SELECT COUNT(*) FROM products p JOIN order_items oi ON p.product_id = oi.product_id WHERE p.supplier_id = v_supplier_id);
                IF v_inactive_supplier_orders > 0 THEN
                    REPEAT
                        DELETE FROM order_items
                        WHERE order_item_id IN (
                            SELECT oi.order_item_id
                            FROM order_items oi
                            JOIN products p ON oi.product_id = p.product_id
                            WHERE p.supplier_id = v_supplier_id
                            LIMIT 1
                        );
                        SET v_inactive_supplier_orders = v_inactive_supplier_orders - 1;
                    UNTIL v_inactive_supplier_orders = 0 END REPEAT;
                END IF;
            END L5;
        END IF;
    END FOR supplier_loop;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, supplier-management
-- EXPECTED: Random product inserts, supplier updates, order item deletes, and discount logic
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 9
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_employee_id INT;
    DECLARE v_employee_commission DOUBLE;
    DECLARE v_employee_total DOUBLE DEFAULT 0.0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Commission calculation error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_total_emp INT;
        SET v_total_emp = (SELECT COUNT(*) FROM employees);
        FOR emp AS SELECT employee_id, employee_commission FROM employees LIMIT 3 DO
            SET v_employee_id = emp.employee_id;
            SET v_employee_commission = emp.employee_commission;
            L1: BEGIN
                DECLARE v_order_count INT;
                SET v_order_count = (SELECT COUNT(*) FROM orders WHERE customer_id = (SELECT MIN(customer_id) FROM customers));
                IF v_order_count > 0 THEN
                    L2: BEGIN
                        DECLARE v_commission_changed BOOLEAN DEFAULT false;
                        WHILE v_employee_commission > 0 AND v_employee_commission < 2000 DO
                            UPDATE employees SET employee_commission = employee_commission + 100.0 WHERE employee_id = v_employee_id;
                            SET v_employee_total = v_employee_total + 100.0;
                            SET v_employee_commission = (SELECT employee_commission FROM employees WHERE employee_id = v_employee_id);
                            SET v_commission_changed = true;
                        END WHILE;
                        IF v_commission_changed THEN
                            REPEAT
                                SET v_employee_total = v_employee_total - 50.0;
                                UPDATE employees SET employee_commission = employee_commission - 50.0 WHERE employee_id = v_employee_id;
                            UNTIL v_employee_total <= 1500 END REPEAT;
                        END IF;
                    END L2;
                END IF;
            END L1;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, WHILE, REPEAT, DML, deep-nesting, exception-handler, commission-logic
-- EXPECTED: Random commission updates based on order logic
-- EXECUTES: FOR, IF, WHILE, REPEAT, nested BEGIN...END, exception handler

-- Script 10
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_category STRING;
    DECLARE v_order_count INT DEFAULT 0;
    DECLARE v_customer_total DOUBLE DEFAULT 0.0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Customer order calculation error'); END;

    -- Wide, deep, random logic, labels
    customer_category_loop: FOR category AS SELECT DISTINCT customer_type FROM customers LIMIT 2 DO
        SET v_customer_category = category.customer_type;
        SET v_order_count = (SELECT COUNT(*) FROM orders o JOIN customers c ON o.customer_id = c.customer_id WHERE c.customer_type = v_customer_category);
        IF v_order_count > 0 THEN
            L1: BEGIN
                DECLARE v_average_order DOUBLE;
                SET v_average_order = (SELECT AVG(o.order_total) FROM orders o JOIN customers c ON o.customer_id = c.customer_id WHERE c.customer_type = v_customer_category);
                L2: BEGIN
                    IF v_average_order > 100 THEN
                        SET v_customer_total = v_customer_total + v_average_order;
                        BEGIN
                            DECLARE v_order_limit INT DEFAULT 3;
                            FOR order_row AS SELECT o.order_id FROM orders o JOIN customers c ON o.customer_id = c.customer_id WHERE c.customer_type = v_customer_category LIMIT v_order_limit DO
                                BEGIN
                                    DECLARE v_order_items_count INT;
                                    SET v_order_items_count = (SELECT COUNT(*) FROM order_items WHERE order_id = order_row.order_id);
                                    IF v_order_items_count < 1 THEN
                                        BEGIN
                                            REPEAT
                                                INSERT INTO order_items (order_item_id, order_id, product_id, item_quantity, item_price_per_unit, item_discount, item_tax, item_total)
                                                VALUES ((SELECT MAX(order_item_id) + 1 FROM order_items), order_row.order_id, (SELECT MIN(product_id) FROM products), 1, 10.0, 0.0, 1.0, 11.0);
                                                SET v_order_items_count = v_order_items_count + 1;
                                            UNTIL v_order_items_count >= 1 END REPEAT;
                                        END;
                                    END IF;
                                END;
                            END FOR;
                        END;
                    ELSE
                        L3: BEGIN
                            REPEAT
                                SET v_customer_total = v_customer_total + 10.0;
                                INSERT INTO orders (order_id, customer_id, order_date, order_status, order_total, order_tax, order_discount, order_shipping)
                                VALUES ((SELECT MAX(order_id) + 1 FROM orders), (SELECT MIN(customer_id) FROM customers WHERE customer_type = v_customer_category), CURRENT_TIMESTAMP, 'Low Value', 10.0, 1.0, 0.0, 2.0);
                                SET v_order_count = v_order_count + 1;
                            UNTIL v_customer_total > 100 END REPEAT;
                        END L3;
                    END IF;
                END L2;
            END L1;
        END IF;
    END FOR customer_category_loop;
    L4: BEGIN
        DECLARE v_min_orders INT DEFAULT 1;
        IF v_order_count < v_min_orders THEN
            BEGIN
                DECLARE v_new_order_id INT;
                SET v_new_order_id = (SELECT MAX(order_id) + 1 FROM orders);
                INSERT INTO orders (order_id, customer_id, order_date, order_status, order_total, order_tax, order_discount, order_shipping)
                VALUES (v_new_order_id, (SELECT MIN(customer_id) FROM customers), CURRENT_TIMESTAMP, 'Auto Insert', 15.0, 1.5, 0.0, 2.5);
            END;
        END IF;
    END L4;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, customer-logic
-- EXPECTED: Random order/item inserts, conditional logic on customer types
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 11
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_max_manager_id INT DEFAULT 2;
    DECLARE v_employee_count INT DEFAULT 0;
    DECLARE v_department_levels INT DEFAULT 0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Employee hierarchy error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_manager_exists BOOLEAN DEFAULT false;
        WHILE v_manager_exists = false AND v_max_manager_id < 4 DO
            SET v_manager_exists = (SELECT COUNT(*) > 0 FROM employees WHERE employee_manager_id = v_max_manager_id);
            IF v_manager_exists THEN
                L1: BEGIN
                    REPEAT
                        SET v_department_levels = v_department_levels + 1;
                        SET v_employee_count = (SELECT COUNT(*) FROM employees WHERE employee_manager_id = v_max_manager_id);
                        WHILE v_employee_count > 0 AND v_employee_count < 10 DO
                            BEGIN
                                DECLARE v_new_id INT;
                                SET v_new_id = (SELECT COALESCE(MAX(employee_id), 0) + 1 FROM employees);
                                INSERT INTO employees VALUES (v_new_id, 'New Emp ' || v_new_id, 'new' || v_new_id || '@company.com', '555-0000', 'Intern', 'Ops', CURRENT_TIMESTAMP, NULL, 30000.0, 0.0, v_max_manager_id, '123 New St', NULL, 'Newtown', 'CA', '00000', 'USA', CURRENT_TIMESTAMP, 'Male', 'Single', 0);
                                SET v_employee_count = v_employee_count + 1;
                            END;
                        END WHILE;
                        SET v_max_manager_id = v_max_manager_id + 1;
                        SET v_manager_exists = (SELECT COUNT(*) > 0 FROM employees WHERE employee_manager_id = v_max_manager_id);
                    UNTIL v_max_manager_id >= 4 OR v_department_levels > 2 END REPEAT;
                END L1;
            ELSE
                BEGIN
                    DECLARE v_temp_id INT;
                    SET v_temp_id = v_max_manager_id + 1;
                    INSERT INTO employees VALUES (v_temp_id, 'Manager ' || v_temp_id, 'manager' || v_temp_id || '@company.com', '555-0000', 'Manager', 'HR', CURRENT_TIMESTAMP, NULL, 60000.0, 0.0, NULL, '123 Mgmt St', NULL, 'Newtown', 'CA', '00000', 'USA', CURRENT_TIMESTAMP, 'Male', 'Married', 1);
                    SET v_manager_exists = true;
                    SET v_max_manager_id = v_temp_id;
                END;
            END IF;
        END WHILE;
    END L0;
    L2: BEGIN
        DECLARE v_low_level_emp INT DEFAULT 0;
        REPEAT
            SET v_low_level_emp = v_low_level_emp + 1;
            INSERT INTO employees VALUES (v_low_level_emp + 10, 'Temp Emp ' || v_low_level_emp, 'temp' || v_low_level_emp || '@company.com', '555-0000', 'Temp', 'Temp', CURRENT_TIMESTAMP, CURRENT_TIMESTAMP, 20000.0, 0.0, NULL, '123 Temp St', NULL, 'Temp City', 'TX', '00000', 'USA', CURRENT_TIMESTAMP, 'Female', 'Single', 0);
        UNTIL v_low_level_emp > 2 END REPEAT;
    END L2;
END;
--QUERY-DELIMITER-END
-- TAGS: WHILE, REPEAT, IF, DML, deep-nesting, exception-handler, employee-management
-- EXPECTED: Random employee inserts, hierarchy construction, loop termination
-- EXECUTES: WHILE, REPEAT, IF, nested BEGIN...END, exception handler

-- Script 12
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_shipping_cost DOUBLE DEFAULT 0.0;
    DECLARE v_shipping_city STRING;
    DECLARE v_shipping_orders INT DEFAULT 0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Shipping analysis error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_max_orders INT DEFAULT 3;
        FOR city AS SELECT DISTINCT order_shipping_city FROM orders LIMIT 2 DO
            SET v_shipping_city = city.order_shipping_city;
            SET v_shipping_cost = (SELECT SUM(order_shipping) FROM orders WHERE order_shipping_city = v_shipping_city);
            IF v_shipping_cost > 5 THEN
                L1: BEGIN
                    REPEAT
                        SET v_shipping_cost = v_shipping_cost - 1.0;
                        UPDATE orders SET order_shipping = order_shipping - 0.5 WHERE order_shipping_city = v_shipping_city;
                    UNTIL v_shipping_cost < 5 END REPEAT;
                END L1;
                BEGIN
                    DECLARE v_customer_in_shipping_city INT;
                    SET v_customer_in_shipping_city = (SELECT COUNT(*) FROM customers c WHERE c.customer_city = v_shipping_city);
                    IF v_customer_in_shipping_city = 0 THEN
                        BEGIN
                            DECLARE v_new_customer_id INT;
                            SET v_new_customer_id = (SELECT MAX(customer_id) + 1 FROM customers);
                            INSERT INTO customers VALUES (
                                v_new_customer_id,
                                'Local Customer',
                                'local@' || v_shipping_city || '.com',
                                '555-0000',
                                '123 Main St',
                                v_shipping_city,
                                (SELECT DISTINCT order_shipping_state FROM orders WHERE order_shipping_city = v_shipping_city LIMIT 1),
                                '00000',
                                'USA',
                                'Retail',
                                CURRENT_TIMESTAMP, 0.0, NULL, NULL, NULL, true, true, 0.0, NULL, NULL
                            );
                        END;
                    END IF;
                END;
            ELSE
                L2: BEGIN
                    SET v_shipment_orders = (SELECT COUNT(*) FROM orders WHERE order_shipping_city = v_shipping_city);
                    WHILE v_shipping_orders < v_max_orders DO
                        BEGIN
                            DECLARE v_new_order_id INT;
                            SET v_new_order_id = (SELECT MAX(order_id) + 1 FROM orders);
                            INSERT INTO orders (
                                order_id, customer_id, order_date, order_status, order_total,
                                order_tax, order_discount, order_shipping, order_shipping_city,
                                order_shipping_state, order_shipping_zip, order_shipping_country
                            ) VALUES (
                                v_new_order_id,
                                (SELECT MIN(customer_id) FROM customers),
                                CURRENT_TIMESTAMP, 'Local Insert', 10.0, 1.0, 0.0, 2.0,
                                v_shipping_city, 'TX', '00000', 'USA'
                            );
                            SET v_shipping_orders = v_shipping_orders + 1;
                        END;
                    END WHILE;
                END L2;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, WHILE, DML, deep-nesting, exception-handler, shipping-logic
-- EXPECTED: Random shipping cost adjustments, customer/order inserts for local cities
-- EXECUTES: FOR, IF, REPEAT, WHILE, nested BEGIN...END, exception handler

-- Script 13
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_supplier_contact_count INT DEFAULT 0;
    DECLARE v_supplier_rating_change DOUBLE DEFAULT 0.0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Supplier contact error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_supplier_count INT;
        SET v_supplier_count = (SELECT COUNT(*) FROM suppliers WHERE supplier_active = true);
        FOR supplier AS SELECT supplier_id, supplier_contact, supplier_rating FROM suppliers WHERE supplier_active = true LIMIT 2 DO
            SET v_supplier_contact_count = v_supplier_contact_count + 1;
            L1: BEGIN
                DECLARE v_supplier_orders INT;
                SET v_supplier_orders = (SELECT COUNT(*) FROM products p JOIN order_items oi ON p.product_id = oi.product_id WHERE p.supplier_id = supplier.supplier_id);
                IF v_supplier_orders > 0 THEN
                    L2: BEGIN
                        SET v_supplier_rating_change = 0.1;
                        REPEAT
                            SET v_supplier_rating_change = v_supplier_rating_change - 0.01;
                            UPDATE suppliers SET supplier_rating = supplier_rating + v_supplier_rating_change WHERE supplier_id = supplier.supplier_id;
                        UNTIL v_supplier_rating_change <= 0.0 END REPEAT;
                    END L2;
                ELSE
                    BEGIN
                        DECLARE v_new_product_id INT;
                        SET v_new_product_id = (SELECT MAX(product_id) + 1 FROM products);
                        INSERT INTO products VALUES (
                            v_new_product_id,
                            'New Supplier Product',
                            'General',
                            39.99,
                            29.99,
                            supplier.supplier_id,
                            supplier.supplier_contact,
                            supplier.supplier_contact,
                            '123 New St',
                            5,
                            'Blue',
                            'M',
                            0.9,
                            'New supplier item',
                            4.0,
                            0,
                            CURRENT_TIMESTAMP,
                            CURRENT_TIMESTAMP,
                            true,
                            0.0,
                            0.19
                        );
                        WHILE supplier.supplier_rating < 4.5 DO
                            UPDATE suppliers SET supplier_rating = supplier_rating + 0.1 WHERE supplier_id = supplier.supplier_id;
                            SET supplier.supplier_rating = (SELECT MAX(supplier_rating) FROM suppliers WHERE supplier_id = supplier.supplier_id);
                        END WHILE;
                    END;
                END IF;
            END L1;
        END FOR;
    END L0;
    BEGIN
        DECLARE v_few_contacts BOOLEAN DEFAULT false;
        SET v_few_contacts = (v_supplier_contact_count < 2);
        IF v_few_contacts THEN
            L3: BEGIN
                DECLARE v_new_contact_id INT;
                SET v_new_contact_id = (SELECT MAX(supplier_id) + 1 FROM suppliers);
                INSERT INTO suppliers VALUES (
                    v_new_contact_id,
                    'Supplier for Contact',
                    'Contact',
                    'Contact',
                    'contact@supplier.com',
                    '555-0000',
                    '123 Contact St',
                    'Contact City',
                    'CA',
                    '00000',
                    'USA',
                    'West',
                    'Contacts only',
                    true
                );
                SET v_supplier_contact_count = v_supplier_contact_count + 1;
            END L3;
        END IF;
    END;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, WHILE, DML, deep-nesting, exception-handler, supplier-contact-logic
-- EXPECTED: Random supplier rating adjustments, product inserts, supplier contact actions
-- EXECUTES: FOR, IF, REPEAT, WHILE, nested BEGIN...END, exception handler

-- Script 14
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_employee_hire_year INT;
    DECLARE v_employee_tenure_years INT;
    DECLARE v_salary_bin INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Tenure analysis error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_emp_count INT DEFAULT 0;
        FOR emp AS SELECT employee_id, employee_hire_date, employee_salary FROM employees LIMIT 3 DO
            SET v_employee_hire_year = YEAR(emp.employee_hire_date);
            SET v_employee_tenure_years = DATEDIFF(YEAR, emp.employee_hire_date, CURRENT_TIMESTAMP);
            SET v_salary_bin = FLOOR(emp.employee_salary / 20000.0) + 1;
            IF v_employee_tenure_years > 1 AND v_employee_tenure_years <= 7 THEN
                L1: BEGIN
                    CASE v_salary_bin
                        WHEN 1 THEN
                            UPDATE employees SET employee_salary = employee_salary * 1.1 WHERE employee_id = emp.employee_id;
                        WHEN 2 THEN
                            BEGIN
                                REPEAT
                                    SET v_salary_bin = v_salary_bin + 1;
                                    SET v_emp_count = v_emp_count + 1;
                                    UPDATE employees SET employee_commission = employee_commission * 1.05 WHERE employee_id = emp.employee_id;
                                UNTIL v_salary_bin > 3 OR v_emp_count > 2 END REPEAT;
                            END;
                        ELSE
                            L2: BEGIN
                                DECLARE v_hire_month INT;
                                SET v_hire_month = MONTH(emp.employee_hire_date);
                                IF v_hire_month % 2 = 0 THEN
                                    BEGIN
                                        SET v_hire_month = v_hire_month + 1;
                                        UPDATE employees SET employee_salary = employee_salary * 0.95 WHERE employee_id = emp.employee_id;
                                    END;
                                END IF;
                            END L2;
                    END CASE;
                END L1;
                SET v_emp_count = v_emp_count + 1;
            ELSE
                L3: BEGIN
                    DECLARE v_year_diff INT;
                    SET v_year_diff = (SELECT DATEDIFF(YEAR, MIN(employee_hire_date), CURRENT_TIMESTAMP) FROM employees);
                    WHILE v_year_diff > 5 DO
                        SET v_year_diff = v_year_diff - 1;
                        INSERT INTO employees VALUES (
                            (SELECT MAX(employee_id) + 1 FROM employees),
                            'Tenure Fill ' || v_year_diff,
                            'tenure' || v_year_diff || '@company.com',
                            '555-0000',
                            'Tenure',
                            'Tenure',
                            CURRENT_TIMESTAMP,
                            NULL,
                            35000.0,
                            0.0,
                            NULL,
                            '123 Tenure St',
                            NULL,
                            'Tenureville',
                            'CA',
                            '00000',
                            'USA',
                            CURRENT_TIMESTAMP,
                            'Male',
                            'Single',
                            0
                        );
                    END WHILE;
                END L3;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, CASE, REPEAT, WHILE, DML, deep-nesting, exception-handler, tenure-logic
-- EXPECTED: Random salary/commission adjustments, employee inserts based on tenure and salary
-- EXECUTES: FOR, IF, CASE, REPEAT, WHILE, nested BEGIN...END, exception handler

-- Script 15
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_source STRING;
    DECLARE v_order_total DOUBLE;
    DECLARE v_order_date TIMESTAMP;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Order source analysis error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_order_source_count INT DEFAULT 0;
        FOR source AS SELECT DISTINCT order_source FROM orders LIMIT 2 DO
            SET v_order_source = source.order_source;
            SET v_order_total = (SELECT SUM(order_total) FROM orders WHERE order_source = source.order_source);
            IF v_order_source = 'Web' OR v_order_source = 'Mobile' THEN
                L1: BEGIN
                    REPEAT
                        SET v_order_source_count = v_order_source_count + 1;
                        SET v_order_date = (SELECT MAX(order_date) FROM orders WHERE order_source = source.order_source);
                        IF v_order_date < DATEADD(DAY, -30, CURRENT_TIMESTAMP) THEN
                            BEGIN
                                INSERT INTO orders (
                                    order_id, customer_id, order_date, order_status, order_total,
                                    order_tax, order_discount, order_shipping, order_source
                                ) VALUES (
                                    (SELECT MAX(order_id) + 1 FROM orders),
                                    (SELECT MIN(customer_id) FROM customers),
                                    CURRENT_TIMESTAMP,
                                    'Recent Web',
                                    50.0,
                                    5.0,
                                    0.0,
                                    5.0,
                                    source.order_source
                                );
                                SET v_order_date = CURRENT_TIMESTAMP;
                            END;
                        END IF;
                    UNTIL v_order_source_count > 1 OR v_order_date > DATEADD(DAY, -30, CURRENT_TIMESTAMP) END REPEAT;
                END L1;
            ELSE
                BEGIN
                    DECLARE v_fake_source STRING DEFAULT CONCAT('Fake ', source.order_source);
                    INSERT INTO orders (
                        order_id, customer_id, order_date, order_status, order_total,
                        order_tax, order_discount, order_shipping, order_source
                    ) VALUES (
                        (SELECT MAX(order_id) + 1 FROM orders),
                        (SELECT MIN(customer_id) FROM customers),
                        CURRENT_TIMESTAMP,
                        'Generated',
                        30.0,
                        3.0,
                        0.0,
                        3.0,
                        v_fake_source
                    );
                END;
            END IF;
        END FOR;
    END L0;
    BEGIN
        DECLARE v_source_total DOUBLE;
        SET v_source_total = (SELECT SUM(order_total) FROM orders WHERE order_source LIKE 'Fake%');
        IF v_source_total < 100 THEN
            REPEAT
                INSERT INTO orders (
                    order_id, customer_id, order_date, order_status, order_total,
                    order_tax, order_discount, order_shipping, order_source
                ) VALUES (
                    (SELECT MAX(order_id) + 1 FROM orders),
                    (SELECT MIN(customer_id) FROM customers),
                    CURRENT_TIMESTAMP,
                    'Fake Source',
                    15.0,
                    1.5,
                    0.0,
                    1.5,
                    'Fake Source'
                );
                SET v_source_total = (SELECT SUM(order_total) FROM orders WHERE order_source LIKE 'Fake%');
            UNTIL v_source_total >= 100 END REPEAT;
        END IF;
    END;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, source-analysis
-- EXPECTED: Random order inserts, source analysis, fake order generation
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 16
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_active BOOLEAN;
    DECLARE v_customer_lifetime_value DOUBLE DEFAULT 0.0;
    DECLARE v_customer_order_count INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Customer profitability error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_target_value DOUBLE DEFAULT 500.0;
        FOR cust AS SELECT customer_id, customer_active_status FROM customers LIMIT 3 DO
            SET v_customer_active = cust.customer_active_status;
            SET v_customer_lifetime_value = (SELECT SUM(order_total) FROM orders WHERE customer_id = cust.customer_id);
            SET v_customer_order_count = (SELECT COUNT(*) FROM orders WHERE customer_id = cust.customer_id);
            IF v_customer_active = true AND v_customer_lifetime_value > 0.0 THEN
                L1: BEGIN
                    IF v_customer_lifetime_value < v_target_value THEN
                        L2: BEGIN
                            REPEAT
                                INSERT INTO orders (
                                    order_id, customer_id, order_date, order_status,
                                    order_total, order_tax, order_discount, order_shipping
                                ) VALUES (
                                    (SELECT MAX(order_id) + 1 FROM orders),
                                    cust.customer_id,
                                    CURRENT_TIMESTAMP,
                                    'LTV Boost',
                                    40.0,
                                    4.0,
                                    0.0,
                                    4.0
                                );
                                SET v_customer_lifetime_value = (SELECT SUM(order_total) FROM orders WHERE customer_id = cust.customer_id);
                            UNTIL v_customer_lifetime_value >= v_target_value OR v_customer_order_count > 5 END REPEAT;
                        END L2;
                    END IF;
                    BEGIN
                        DECLARE v_discount_to_apply DOUBLE;
                        CASE v_customer_order_count
                            WHEN 1 THEN SET v_discount_to_apply = 0.10;
                            WHEN 2 THEN SET v_discount_to_apply = 0.05;
                            ELSE SET v_discount_to_apply = 0.0;
                        END CASE;
                        IF v_discount_to_apply > 0 THEN
                            L3: BEGIN
                                REPEAT
                                    UPDATE customers SET customer_discount_percentage = v_discount_to_apply WHERE customer_id = cust.customer_id;
                                    SET v_discount_to_apply = v_discount_to_apply - 0.01;
                                UNTIL v_discount_to_apply <= 0.0 END REPEAT;
                            END L3;
                        END IF;
                    END;
                END L1;
            ELSE
                BEGIN
                    DECLARE v_inactive_customer_count INT;
                    SET v_inactive_customer_count = (SELECT COUNT(*) FROM customers WHERE customer_active_status = false);
                    IF v_inactive_customer_count > 0 THEN
                        L4: BEGIN
                            WHILE v_inactive_customer_count > 0 AND v_customer_lifetime_value < 200 DO
                                SET v_inactive_customer_count = v_inactive_customer_count - 1;
                                INSERT INTO orders (
                                    order_id, customer_id, order_date, order_status,
                                    order_total, order_tax, order_discount, order_shipping
                                ) VALUES (
                                    (SELECT MAX(order_id) + 1 FROM orders),
                                    (SELECT MIN(customer_id) FROM customers WHERE customer_active_status = false),
                                    CURRENT_TIMESTAMP,
                                    'Reactivation',
                                    20.0,
                                    2.0,
                                    0.0,
                                    2.0
                                );
                                SET v_customer_lifetime_value = (SELECT SUM(order_total) FROM orders WHERE customer_id = cust.customer_id);
                            END WHILE;
                        END L4;
                    END IF;
                END;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, CASE, REPEAT, WHILE, DML, deep-nesting, exception-handler, customer-lifecycle
-- EXPECTED: Random order inserts, discount updates, customer lifecycle manipulation
-- EXECUTES: FOR, IF, CASE, REPEAT, WHILE, nested BEGIN...END, exception handler

-- Script 17
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_supplier_region_count INT DEFAULT 0;
    DECLARE v_supplier_rating_change DOUBLE DEFAULT 0.0;
    DECLARE v_region_to_boost STRING;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Supplier region analysis error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_supplier_count INT DEFAULT 2;
        FOR region AS SELECT DISTINCT supplier_region FROM suppliers LIMIT 2 DO
            SET v_region_to_boost = region.supplier_region;
            SET v_supplier_region_count = (SELECT COUNT(*) FROM suppliers WHERE supplier_region = v_region_to_boost);
            IF v_supplier_region_count > v_min_supplier_count THEN
                L1: BEGIN
                    REPEAT
                        SET v_supplier_rating_change = 0.05;
                        REPEAT
                            UPDATE suppliers SET supplier_rating = supplier_rating + v_supplier_rating_change WHERE supplier_region = v_region_to_boost;
                            SET v_supplier_rating_change = v_supplier_rating_change - 0.01;
                        UNTIL v_supplier_rating_change <= 0.0 END REPEAT;
                        SET v_min_supplier_count = v_min_supplier_count + 1;
                    UNTIL v_min_supplier_count > 4 END REPEAT;
                END L1;
                BEGIN
                    DECLARE v_products_in_region INT;
                    SET v_products_in_region = (SELECT COUNT(*) FROM products p JOIN suppliers s ON p.supplier_id = s.supplier_id WHERE s.supplier_region = v_region_to_boost);
                    IF v_products_in_region < 3 THEN
                        REPEAT
                            INSERT INTO products VALUES (
                                (SELECT MAX(product_id) + 1 FROM products),
                                'Region Product',
                                'General',
                                39.99,
                                29.99,
                                (SELECT MIN(supplier_id) FROM suppliers WHERE supplier_region = v_region_to_boost),
                                'Region Contact',
                                'Region Role',
                                '123 Main St',
                                10,
                                'Blue',
                                'M',
                                1.0,
                                'Region product',
                                3.5,
                                0,
                                CURRENT_TIMESTAMP,
                                CURRENT_TIMESTAMP,
                                true,
                                0.0,
                                0.19
                            );
                        UNTIL (SELECT COUNT(*) FROM products p JOIN suppliers s ON p.supplier_id = s.supplier_id WHERE s.supplier_region = v_region_to_boost) >= 3 END REPEAT;
                    END IF;
                END;
            ELSE
                L2: BEGIN
                    DECLARE v_new_supplier_id INT;
                    SET v_new_supplier_id = (SELECT MAX(supplier_id) + 1 FROM suppliers);
                    INSERT INTO suppliers VALUES (
                        v_new_supplier_id,
                        'Region ' || v_region_to_boost,
                        'Regional Contact',
                        'Regional Contact',
                        'contact@regional.com',
                        '555-0000',
                        '123 Regional St',
                        'Regional City',
                        'CA',
                        '00000',
                        'USA',
                        v_region_to_boost,
                        'Regional',
                        true
                    );
                    SET v_supplier_region_count = v_supplier_region_count + 1;
                END L2;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, region-analysis
-- EXPECTED: Random supplier rating updates, product inserts, supplier inserts as needed
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 18
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_employee_birth_year INT;
    DECLARE v_employee_age INT;
    DECLARE v_employee_city STRING;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Employee demographics error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_emp_count INT DEFAULT 0;
        FOR emp AS SELECT employee_id, employee_birth_date, employee_city FROM employees LIMIT 3 DO
            SET v_employee_birth_year = YEAR(emp.employee_birth_date);
            SET v_employee_age = DATEDIFF(YEAR, emp.employee_birth_date, CURRENT_TIMESTAMP);
            SET v_employee_city = emp.employee_city;
            IF v_employee_age < 30 THEN
                L1: BEGIN
                    DECLARE v_city_emp_count INT;
                    SET v_city_emp_count = (SELECT COUNT(*) FROM employees WHERE employee_city = v_employee_city);
                    IF v_city_emp_count < 2 THEN
                        REPEAT
                            SET v_emp_count = v_emp_count + 1;
                            INSERT INTO employees VALUES (
                                (SELECT MAX(employee_id) + 1 FROM employees),
                                'Young Emp ' || v_emp_count,
                                'emp' || v_emp_count || '@company.com',
                                '555-0000',
                                'Junior',
                                'Ops',
                                CURRENT_TIMESTAMP,
                                NULL,
                                35000.0,
                                0.0,
                                NULL,
                                '123 Jun St',
                                NULL,
                                v_employee_city,
                                (SELECT DISTINCT employee_state FROM employees WHERE employee_city = v_employee_city LIMIT 1),
                                '00000',
                                'USA',
                                DATEADD(YEAR, -(FLOOR(RAND() * 10) + 20), CURRENT_TIMESTAMP),
                                'Male',
                                'Single',
                                0
                            );
                        UNTIL v_emp_count > 1 END REPEAT;
                    END IF;
                END L1;
            ELSE
                BEGIN
                    DECLARE v_birth_month INT;
                    SET v_birth_month = MONTH(emp.employee_birth_date);
                    CASE v_birth_month
                        WHEN 1 THEN
                            UPDATE employees SET employee_commission = employee_commission + 100.0 WHERE employee_id = emp.employee_id;
                        WHEN 6 THEN
                            BEGIN
                                WHILE v_employee_age < 40 DO
                                    SET v_employee_age = v_employee_age + 1;
                                    INSERT INTO customers VALUES (
                                        (SELECT MAX(customer_id) + 1 FROM customers),
                                        'Age Group Customer',
                                        'age@group.com',
                                        '555-0001',
                                        '123 Age St',
                                        v_employee_city,
                                        (SELECT DISTINCT employee_state FROM employees WHERE employee_city = v_employee_city LIMIT 1),
                                        '00000',
                                        'USA',
                                        'Senior',
                                        CURRENT_TIMESTAMP,
                                        0.0,
                                        NULL,
                                        NULL,
                                        NULL,
                                        true,
                                        true,
                                        0.0,
                                        NULL,
                                        NULL
                                    );
                                END WHILE;
                            END;
                        ELSE
                            L2: BEGIN
                                DECLARE v_new_birth_year INT;
                                SET v_new_birth_year = v_employee_birth_year - (FLOOR(RAND() * 5) + 1);
                                INSERT INTO employees VALUES (
                                    (SELECT MAX(employee_id) + 1 FROM employees),
                                    'Filler Emp',
                                    'filler@company.com',
                                    '555-0000',
                                    'Filler',
                                    'Ops',
                                    CURRENT_TIMESTAMP,
                                    NULL,
                                    30000.0,
                                    0.0,
                                    NULL,
                                    '123 Filler St',
                                    NULL,
                                    v_employee_city,
                                    (SELECT DISTINCT employee_state FROM employees WHERE employee_city = v_employee_city LIMIT 1),
                                    '00000',
                                    'USA',
                                    MAKE_DATE(v_new_birth_year, MONTH(CURRENT_TIMESTAMP), DAY(CURRENT_TIMESTAMP)),
                                    'Female',
                                    'Single',
                                    0
                                );
                            END L2;
                    END CASE;
                END;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, CASE, REPEAT, WHILE, DML, deep-nesting, exception-handler, demographics
-- EXPECTED: Random employee/customer inserts, commission updates, age-based logic
-- EXECUTES: FOR, IF, CASE, REPEAT, WHILE, nested BEGIN...END, exception handlers

-- Script 19
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_launch_year INT;
    DECLARE v_active_product_count INT DEFAULT 0;
    DECLARE v_new_composite_id INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Product cohort error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR product AS SELECT product_id, product_launch_date, product_name, product_is_active FROM products LIMIT 3 DO
            SET v_product_launch_year = YEAR(product.product_launch_date);
            L1: BEGIN
                IF product.product_is_active = true THEN
                    SET v_active_product_count = v_active_product_count + 1;
                    IF v_product_launch_year < 2023 THEN
                        L2: BEGIN
                            REPEAT
                                SET v_new_composite_id = product.product_id * 100 + (SELECT COUNT(*) FROM products WHERE YEAR(product_launch_date) = v_product_launch_year);
                                INSERT INTO order_items (
                                    order_item_id,
                                    order_id,
                                    product_id,
                                    item_quantity,
                                    item_price_per_unit,
                                    item_discount,
                                    item_tax,
                                    item_total
                                ) VALUES (
                                    v_new_composite_id,
                                    (SELECT MIN(order_id) FROM orders),
                                    product.product_id,
                                    1,
                                    (SELECT product_price FROM products WHERE product_id = product.product_id),
                                    0.0,
                                    0.0,
                                    (SELECT product_price FROM products WHERE product_id = product.product_id)
                                );
                                SET v_active_product_count = v_active_product_count - 1;
                            UNTIL v_active_product_count < 1 END REPEAT;
                        END L2;
                    END IF;
                ELSE
                    BEGIN
                        DECLARE v_launch_month INT;
                        SET v_launch_month = MONTH(product.product_launch_date);
                        WHILE v_launch_month < 12 DO
                            SET v_launch_month = v_launch_month + 1;
                            BEGIN
                                DECLARE v_new_product_id INT;
                                SET v_new_product_id = (SELECT MAX(product_id) + 1 FROM products);
                                INSERT INTO products VALUES (
                                    v_new_product_id,
                                    'Covert ' || product.product_name,
                                    'Specialty',
                                    49.99,
                                    34.99,
                                    (SELECT MIN(supplier_id) FROM suppliers),
                                    'Stealth Contact',
                                    'Cloaked Role',
                                    '123 Cloak St',
                                    2,
                                    'Black',
                                    'Medium',
                                    1.1,
                                    'Hidden product',
                                    0.0,
                                    0,
                                    MAKE_DATE(v_product_launch_year, v_launch_month, 1),
                                    CURRENT_TIMESTAMP,
                                    true,
                                    0.0,
                                    0.21
                                );
                            END;
                        END WHILE;
                    END;
                END IF;
            END L1;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, WHILE, DML, deep-nesting, exception-handler, product-cohort
-- EXPECTED: Random order item and product inserts, launch cohort logic
-- EXECUTES: FOR, IF, REPEAT, WHILE, nested BEGIN...END, exception handler

-- Script 20
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_cost_change DOUBLE DEFAULT 0.0;
    DECLARE v_last_update_days INT;
    DECLARE v_product_is_active BOOLEAN;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Cost adjustment error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR product AS SELECT product_id, product_cost, product_last_update, product_is_active FROM products LIMIT 3 DO
            SET v_last_update_days = DATEDIFF(DAY, product.product_last_update, CURRENT_TIMESTAMP);
            SET v_product_is_active = product.product_is_active;
            L1: BEGIN
                IF v_product_is_active = true AND v_last_update_days > 60 THEN
                    L2: BEGIN
                        REPEAT
                            SET v_product_cost_change = v_product_cost_change + 0.05;
                            UPDATE products SET product_cost = product_cost * (1.0 + v_product_cost_change), product_last_update = CURRENT_TIMESTAMP WHERE product_id = product.product_id;
                            SET v_last_update_days = DATEDIFF(DAY, product.product_last_update, CURRENT_TIMESTAMP);
                        UNTIL v_product_cost_change > 0.15 OR v_last_update_days < 10 END REPEAT;
                    END L2;
                ELSE
                    BEGIN
                        DECLARE v_min_cost DOUBLE;
                        SET v_min_cost = (SELECT MIN(product_cost) FROM products WHERE product_is_active = true);
                        IF v_min_cost > 10 THEN
                            L3: BEGIN
                                REPEAT
                                    SET v_product_cost_change = v_product_cost_change - 0.02;
                                    UPDATE products SET product_cost = product_cost * (1.0 + v_product_cost_change) WHERE product_id = product.product_id;
                                UNTIL v_product_cost_change < -0.10 OR (SELECT MIN(product_cost) FROM products WHERE product_is_active = true) <= 10 END REPEAT;
                            END L3;
                        END IF;
                        IF v_product_is_active = false THEN
                            BEGIN
                                DECLARE v_active_product_count INT;
                                SET v_active_product_count = (SELECT COUNT(*) FROM products WHERE product_is_active = true);
                                IF v_active_product_count < 5 THEN
                                    UPDATE products SET product_is_active = true WHERE product_id = (SELECT MIN(product_id) FROM products WHERE product_is_active = false);
                                END IF;
                            END;
                        END IF;
                    END;
                END IF;
            END L1;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, cost-logic
-- EXPECTED: Random cost and status updates, last update refresh, activation logic
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 21
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_shipping_cost DOUBLE;
    DECLARE v_region STRING;
    DECLARE v_region_shipping_changed BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Shipping region logic error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_order_limit INT DEFAULT 3;
        FOR region AS SELECT DISTINCT order_shipping_region FROM orders LIMIT 2 DO
            SET v_region = region.order_shipping_region;
            SET v_order_shipping_cost = (SELECT SUM(order_shipping) FROM orders WHERE order_shipping_region = v_region);
            IF v_order_shipping_cost > 10 THEN
                L1: BEGIN
                    REPEAT
                        SET v_order_shipping_cost = v_order_shipping_cost - 1.0;
                        SET v_region_shipping_changed = true;
                        UPDATE orders SET order_shipping = order_shipping - 0.5 WHERE order_shipping_region = v_region;
                    UNTIL v_order_shipping_cost < 10 END REPEAT;
                END L1;
            END IF;
            BEGIN
                IF v_region_shipping_changed THEN
                    L2: BEGIN
                        DECLARE v_supplier_in_region INT;
                        SET v_supplier_in_region = (SELECT COUNT(*) FROM suppliers WHERE supplier_region = v_region);
                        IF v_supplier_in_region = 0 THEN
                            INSERT INTO suppliers VALUES (
                                (SELECT MAX(supplier_id) + 1 FROM suppliers),
                                'Region Supplier',
                                'Region Contact',
                                'Region Manager',
                                'region@supplier.com',
                                '555-0000',
                                '123 Region St',
                                'Region City',
                                'CA',
                                '00000',
                                'USA',
                                v_region,
                                'General',
                                true
                            );
                        END IF;
                    END L2;
                END IF;
            END;
        END FOR;
    END L0;
    BEGIN
        DECLARE v_nonexistent_orders INT;
        SET v_nonexistent_orders = (SELECT COUNT(*) FROM orders o LEFT JOIN customers c ON o.customer_id = c.customer_id WHERE c.customer_id IS NULL);
        IF v_nonexistent_orders > 0 THEN
            L3: BEGIN
                WHILE v_nonexistent_orders > 0 DO
                    DELETE FROM orders WHERE order_id = (SELECT MIN(order_id) FROM orders o LEFT JOIN customers c ON o.customer_id = c.customer_id WHERE c.customer_id IS NULL);
                    SET v_nonexistent_orders = v_nonexistent_orders - 1;
                END WHILE;
            END L3;
        END IF;
    END;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, WHILE, DML, deep-nesting, exception-handler, shipping-region
-- EXPECTED: Random shipping cost adjustments, supplier inserts, orphaned order cleanup
-- EXECUTES: FOR, IF, REPEAT, WHILE, nested BEGIN...END, exception handler

-- Script 22
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_color STRING;
    DECLARE v_color_count INT;
    DECLARE v_max_price_flag BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Product color error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_products_per_color INT DEFAULT 2;
        FOR color AS SELECT DISTINCT product_color FROM products LIMIT 3 DO
            SET v_product_color = color.product_color;
            SET v_color_count = (SELECT COUNT(*) FROM products WHERE product_color = v_product_color);
            IF v_color_count < v_min_products_per_color THEN
                REPEAT
                    INSERT INTO products VALUES (
                        (SELECT MAX(product_id) + 1 FROM products),
                        'Extra ' || v_product_color,
                        'Accessory',
                        19.99,
                        12.99,
                        (SELECT MIN(supplier_id) FROM suppliers),
                        'Color Contact',
                        'Color Role',
                        '123 Color St',
                        5,
                        v_product_color,
                        'Small',
                        0.5,
                        'Extra color',
                        4.0,
                        0,
                        CURRENT_TIMESTAMP,
                        CURRENT_TIMESTAMP,
                        true,
                        0.0,
                        0.19
                    );
                    SET v_color_count = v_color_count + 1;
                UNTIL v_color_count >= v_min_products_per_color END REPEAT;
            END IF;
            BEGIN
                DECLARE v_max_color_price DOUBLE;
                SET v_max_color_price = (SELECT MAX(product_price) FROM products WHERE product_color = v_product_color);
                IF v_max_color_price > 100 THEN
                    SET v_max_price_flag = true;
                    UPDATE products SET product_price = product_price * 0.85 WHERE product_color = v_product_color AND product_price > 100;
                END IF;
            END;
            IF v_max_price_flag THEN
                BEGIN
                    DECLARE v_products_updated INT;
                    SET v_products_updated = (SELECT COUNT(*) FROM products WHERE product_color = v_product_color AND product_price < 50);
                    IF v_products_updated = 0 THEN
                        REPEAT
                            INSERT INTO products VALUES (
                                (SELECT MAX(product_id) + 1 FROM products),
                                'Low Price ' || v_product_color,
                                'Budget',
                                29.99,
                                19.99,
                                (SELECT MIN(supplier_id) FROM suppliers),
                                'Budget Contact',
                                'Budget Role',
                                '123 Budget St',
                                10,
                                v_product_color,
                                'Medium',
                                0.8,
                                'Low price option',
                                3.0,
                                0,
                                CURRENT_TIMESTAMP,
                                CURRENT_TIMESTAMP,
                                true,
                                0.05,
                                0.19
                            );
                        UNTIL (SELECT COUNT(*) FROM products WHERE product_color = v_product_color AND product_price < 50) > 0 END REPEAT;
                    END IF;
                END;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, product-color
-- EXPECTED: Random product inserts, price adjustments, ensures color diversity
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 23
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_total_tax DOUBLE;
    DECLARE v_order_date_month INT;
    DECLARE v_status_changed BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Order tax logic error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_tax_month DOUBLE DEFAULT 100.0;
        FOR order_row AS SELECT order_id, order_total, order_tax, order_date, order_status FROM orders LIMIT 3 DO
            SET v_order_total_tax = order_row.order_total + order_row.order_tax;
            SET v_order_date_month = MONTH(order_row.order_date);
            IF v_order_date_month % 2 = 0 THEN
                L1: BEGIN
                    IF order_row.order_status = 'Completed' THEN
                        SET v_status_changed = true;
                        UPDATE orders SET order_status = 'Closed' WHERE order_id = order_row.order_id;
                    END IF;
                    BEGIN
                        DECLARE v_adj_tax DOUBLE;
                        SET v_adj_tax = order_row.order_tax * 1.1;
                        SET v_order_total_tax = v_order_total_tax + (v_adj_tax - order_row.order_tax);
                        UPDATE orders SET order_tax = v_adj_tax WHERE order_id = order_row.order_id;
                        IF v_order_total_tax > v_min_tax_month THEN
                            L2: BEGIN
                                REPEAT
                                    SET v_min_tax_month = v_min_tax_month + 10.0;
                                UNTIL v_min_tax_month > v_order_total_tax END REPEAT;
                            END L2;
                        END IF;
                    END;
                END L1;
            ELSE
                L3: BEGIN
                    DECLARE v_order_items_added INT DEFAULT 0;
                    WHILE v_order_items_added < 2 DO
                        INSERT INTO order_items (
                            order_item_id,
                            order_id,
                            product_id,
                            item_quantity,
                            item_price_per_unit,
                            item_discount,
                            item_tax,
                            item_total
                        ) VALUES (
                            (SELECT MAX(order_item_id) + 1 FROM order_items),
                            order_row.order_id,
                            (SELECT MIN(product_id) FROM products),
                            1,
                            10.0,
                            0.0,
                            1.0,
                            11.0
                        );
                        SET v_order_items_added = v_order_items_added + 1;
                    END WHILE;
                END L3;
            END IF;
        END FOR;
    END L0;
    BEGIN
        DECLARE v_completed_orders INT;
        SET v_completed_orders = (SELECT COUNT(*) FROM orders WHERE order_status = 'Completed');
        IF v_completed_orders = 0 THEN
            L4: BEGIN
                REPEAT
                    INSERT INTO orders (
                        order_id,
                        customer_id,
                        order_date,
                        order_status,
                        order_total,
                        order_tax,
                        order_discount,
                        order_shipping
                    ) VALUES (
                        (SELECT MAX(order_id) + 1 FROM orders),
                        (SELECT MIN(customer_id) FROM customers),
                        CURRENT_TIMESTAMP,
                        'Completed',
                        50.0,
                        5.0,
                        0.0,
                        5.0
                    );
                UNTIL (SELECT COUNT(*) FROM orders WHERE order_status = 'Completed') > 1 END REPEAT;
            END L4;
        END IF;
    END;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, WHILE, DML, deep-nesting, exception-handler, tax-logic
-- EXPECTED: Random tax increases, item additions, status updates, order completions
-- EXECUTES: FOR, IF, REPEAT, WHILE, nested BEGIN...END, exception handler

-- Script 24
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_size STRING;
    DECLARE v_size_count INT;
    DECLARE v_min_size_count INT DEFAULT 2;
    DECLARE v_size_has_price_flag BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Product size error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR size AS SELECT DISTINCT product_size FROM products LIMIT 3 DO
            SET v_product_size = size.product_size;
            SET v_size_count = (SELECT COUNT(*) FROM products WHERE product_size = v_product_size);
            IF v_size_count < v_min_size_count THEN
                L1: BEGIN
                    REPEAT
                        INSERT INTO products VALUES (
                            (SELECT MAX(product_id) + 1 FROM products),
                            'Extra ' || v_product_size,
                            'General',
                            24.99,
                            15.99,
                            (SELECT MIN(supplier_id) FROM suppliers),
                            'Size Contact',
                            'Size Role',
                            '123 Size St',
                            5,
                            'Black',
                            v_product_size,
                            0.6,
                            'Extra ' || v_product_size || ' product',
                            3.5,
                            8,
                            CURRENT_TIMESTAMP,
                            CURRENT_TIMESTAMP,
                            true,
                            0.0,
                            0.19
                        );
                        SET v_size_count = v_size_count + 1;
                        SET v_size_has_price_flag = true;
                    UNTIL v_size_count >= v_min_size_count END REPEAT;
                END L1;
            END IF;
            IF v_size_has_price_flag THEN
                BEGIN
                    DECLARE v_avg_size_price DOUBLE;
                    SET v_avg_size_price = (SELECT AVG(product_price) FROM products WHERE product_size = v_product_size);
                    IF v_avg_size_price > 30 THEN
                        REPEAT
                            UPDATE products SET product_discount = product_discount + 0.05 WHERE product_size = v_product_size AND product_discount < 0.15;
                            SET v_avg_size_price = (SELECT AVG(product_price) FROM products WHERE product_size = v_product_size);
                        UNTIL v_avg_size_price <= 30 END REPEAT;
                    END IF;
                END;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, product-size
-- EXPECTED: Random product inserts, discount updates, ensures size diversity
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 25
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_department STRING;
    DECLARE v_dept_emp_count INT;
    DECLARE v_dept_avg_salary DOUBLE;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Department error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR dept AS SELECT DISTINCT employee_department FROM employees LIMIT 3 DO
                    SET v_department = dept.employee_department;
SET v_dept_emp_count = (SELECT COUNT(*) FROM employees WHERE employee_department = v_department);
            SET v_dept_avg_salary = (SELECT AVG(employee_salary) FROM employees WHERE employee_department = v_department);
            IF v_dept_emp_count < 2 THEN
                L1: BEGIN
                    REPEAT
                        INSERT INTO employees VALUES (
                            (SELECT MAX(employee_id) + 1 FROM employees),
                            v_department || ' Emp ' || v_dept_emp_count,
                            'emp' || v_dept_emp_count || '@' || LOWER(v_department) || '.com',
                            '555-0000',
                            'Member',
                            v_department,
                            CURRENT_TIMESTAMP,
                            NULL,
                            35000.0,
                            0.0,
                            NULL,
                            '123 Dept St',
                            NULL,
                            'Dept City',
                            'CA',
                            '00000',
                            'USA',
                            CURRENT_TIMESTAMP,
                            'Female',
                            'Single',
                            0
                        );
                        SET v_dept_emp_count = v_dept_emp_count + 1;
                    UNTIL v_dept_emp_count >= 2 END REPEAT;
END L1;
END IF;
            IF v_dept_avg_salary > 40000 AND v_dept_emp_count >= 2 THEN
BEGIN
                    DECLARE v_salary_decrease DOUBLE;
                    SET v_salary_decrease = 0.95;
                    REPEAT
UPDATE employees SET employee_salary = employee_salary * v_salary_decrease WHERE employee_department = v_department;
SET v_dept_avg_salary = (SELECT AVG(employee_salary) FROM employees WHERE employee_department = v_department);
                    UNTIL v_dept_avg_salary <= 40000 END REPEAT;
END;
            ELSEIF v_dept_avg_salary < 30000 AND v_dept_emp_count >= 2 THEN
                REPEAT
UPDATE employees SET employee_salary = employee_salary * 1.05 WHERE employee_department = v_department AND employee_salary < 30000;
SET v_dept_avg_salary = (SELECT AVG(employee_salary) FROM employees WHERE employee_department = v_department);
                UNTIL v_dept_avg_salary >= 30000 OR v_dept_avg_salary > 50000 END REPEAT;
END IF;
END FOR;
END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, department-logic
-- EXPECTED: Random employee inserts, salary adjustments, department balancing
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 26
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_rating DOUBLE;
    DECLARE v_max_reviews INT DEFAULT 100;
    DECLARE v_min_reviews_flag BOOLEAN DEFAULT true;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Product review analysis error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR product AS SELECT product_id, product_rating, product_reviews FROM products LIMIT 3 DO
            SET v_product_rating = product.product_rating;
            IF product.product_reviews < 10 THEN
                REPEAT
                    SET v_min_reviews_flag = true;
                    UPDATE products SET product_reviews = product_reviews + 5, product_rating = LEAST(product_rating + 0.1, 5.0) WHERE product_id = product.product_id;
                UNTIL product.product_reviews >= 10 END REPEAT;
            END IF;
            BEGIN
                IF v_min_reviews_flag = true AND v_product_rating < 4.5 THEN
                    REPEAT
                        SET v_product_rating = (SELECT product_rating FROM products WHERE product_id = product.product_id);
                        UPDATE products SET product_rating = product_rating + 0.1 WHERE product_id = product.product_id;
                    UNTIL (SELECT product_rating FROM products WHERE product_id = product.product_id) >= 4.5 END REPEAT;
                END IF;
            END;
            IF product.product_rating > 4.5 AND product.product_reviews > v_max_reviews THEN
                BEGIN
                    REPEAT
                        UPDATE products SET product_reviews = product_reviews + 1 WHERE product_id = product.product_id;
                        SET v_max_reviews = v_max_reviews + 1;
                    UNTIL product.product_reviews > v_max_reviews * 2 END REPEAT;
                END;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, review-analysis
-- EXPECTED: Random reviews and rating updates, rating boosting, review inflation
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 27
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_shipping_city STRING;
    DECLARE v_shipping_fee_min DOUBLE DEFAULT 2.0;
    DECLARE v_order_volume INT DEFAULT 0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('City shipping error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR city AS SELECT DISTINCT order_shipping_city FROM orders LIMIT 3 DO
            SET v_order_shipping_city = city.order_shipping_city;
            SET v_order_volume = (SELECT COUNT(*) FROM orders WHERE order_shipping_city = v_order_shipping_city);
            IF v_order_volume > 1 THEN
                L1: BEGIN
                    REPEAT
                        SET v_shipping_fee_min = v_shipping_fee_min + 0.1;
                        UPDATE orders SET order_shipping = order_shipping + 0.2 WHERE order_shipping_city = v_order_shipping_city AND order_shipping < v_shipping_fee_min;
                    UNTIL v_shipping_fee_min > 5.0 END REPEAT;
                END L1;
                BEGIN
                    DECLARE v_order_tax_avg DOUBLE;
                    SET v_order_tax_avg = (SELECT AVG(order_tax) FROM orders WHERE order_shipping_city = v_order_shipping_city);
                    IF v_order_tax_avg > 10 THEN
                        L2: BEGIN
                            REPEAT
                                SET v_order_tax_avg = v_order_tax_avg - 1.0;
                                UPDATE orders SET order_tax = order_tax * 0.9 WHERE order_shipping_city = v_order_shipping_city;
                            UNTIL v_order_tax_avg <= 10 END REPEAT;
                        END L2;
                    END IF;
                END;
            ELSE
                L3: BEGIN
                    REPEAT
                        INSERT INTO orders (
                            order_id,
                            customer_id,
                            order_date,
                            order_status,
                            order_total,
                            order_tax,
                            order_discount,
                            order_shipping,
                            order_shipping_city
                        ) VALUES (
                            (SELECT MAX(order_id) + 1 FROM orders),
                            (SELECT MIN(customer_id) FROM customers),
                            CURRENT_TIMESTAMP,
                            'Volume Insert',
                            12.0,
                            1.2,
                            0.0,
                            2.0,
                            v_order_shipping_city
                        );
                        SET v_order_volume = (SELECT COUNT(*) FROM orders WHERE order_shipping_city = v_order_shipping_city);
                    UNTIL v_order_volume > 1 END REPEAT;
                END L3;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, city-shipping
-- EXPECTED: Random shipping fee and tax updates, order volume manipulation
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 28
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_employee_salary DOUBLE;
    DECLARE v_employee_manager_id INT;
    DECLARE v_manager_has_subordinates BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Employee hierarchy error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR emp AS SELECT employee_id, employee_salary, employee_manager_id FROM employees LIMIT 3 DO
            SET v_employee_salary = emp.employee_salary;
            SET v_employee_manager_id = emp.employee_manager_id;
            IF v_employee_manager_id IS NOT NULL THEN
                L1: BEGIN
                    SET v_manager_has_subordinates = (SELECT COUNT(*) > 0 FROM employees WHERE employee_manager_id = v_employee_manager_id);
                    IF v_manager_has_subordinates THEN
                        L2: BEGIN
                            BEGIN
                                DECLARE v_min_salary DOUBLE DEFAULT 0.0;
                                SET v_min_salary = (SELECT MIN(employee_salary) FROM employees WHERE employee_manager_id = v_employee_manager_id);
                                WHILE v_min_salary < 30000 DO
                                    UPDATE employees SET employee_salary = employee_salary + 1000.0 WHERE employee_id = emp.employee_id;
                                    SET v_employee_salary = v_employee_salary + 1000.0;
                                    SET v_min_salary = (SELECT MIN(employee_salary) FROM employees WHERE employee_manager_id = v_employee_manager_id);
                                END WHILE;
                            END;
                        END L2;
                    ELSE
                        L3: BEGIN
                            REPEAT
                                SET v_employee_salary = v_employee_salary * 1.01;
                                UPDATE employees SET employee_salary = v_employee_salary WHERE employee_id = emp.employee_id;
                            UNTIL (SELECT COUNT(*) FROM employees WHERE employee_manager_id = emp.employee_id) > 0 OR v_employee_salary > 50000 END REPEAT;
                        END L3;
                    END IF;
                END L1;
            ELSE
                BEGIN
                    DECLARE v_lowest_salary DOUBLE;
                    SET v_lowest_salary = (SELECT MIN(employee_salary) FROM employees WHERE employee_manager_id IS NULL);
                    IF v_lowest_salary < 40000 THEN
                        REPEAT
                            INSERT INTO employees VALUES (
                                (SELECT MAX(employee_id) + 1 FROM employees),
                                'Subordinate_' || v_lowest_salary,
                                'subord' || v_lowest_salary || '@company.com',
                                '555-0000',
                                'Subordinate',
                                'HR',
                                CURRENT_TIMESTAMP,
                                NULL,
                                30000.0,
                                0.0,
                                emp.employee_id,
                                '123 Sub St',
                                NULL,
                                'Sub City',
                                'CA',
                                '00000',
                                'USA',
                                CURRENT_TIMESTAMP,
                                'Male',
                                'Single',
                                0
                            );
                            SET v_manager_has_subordinates = (SELECT COUNT(*) > 0 FROM employees WHERE employee_manager_id = emp.employee_id);
                        UNTIL v_manager_has_subordinates OR v_lowest_salary > 35000 END REPEAT;
                    END IF;
                END;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, WHILE, REPEAT, DML, deep-nesting, exception-handler, hierarchy-logic
-- EXPECTED: Random salary adjustments, subordinate creation, manager checks
-- EXECUTES: FOR, IF, WHILE, REPEAT, nested BEGIN...END, exception handler

-- Script 29
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_creation_date TIMESTAMP;
    DECLARE v_order_status STRING;
    DECLARE v_order_days_pending INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Order backlog error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR order_row AS SELECT order_id, order_date, order_status FROM orders LIMIT 3 DO
            SET v_order_creation_date = order_row.order_date;
            SET v_order_status = order_row.order_status;
            SET v_order_days_pending = DATEDIFF(DAY, v_order_creation_date, CURRENT_TIMESTAMP);
            IF v_order_days_pending > 5 AND v_order_status = 'Processing' THEN
                L1: BEGIN
                    REPEAT
                        UPDATE orders SET order_status = 'Pending', order_shipping = order_shipping + 1.0 WHERE order_id = order_row.order_id;
                        SET v_order_creation_date = (SELECT order_date FROM orders WHERE order_id = order_row.order_id);
                        SET v_order_days_pending = DATEDIFF(DAY, v_order_creation_date, CURRENT_TIMESTAMP);
                    UNTIL v_order_days_pending < 3 END REPEAT;
                END L1;
            ELSEIF v_order_days_pending > 30 AND v_order_status = 'Pending' THEN
                L2: BEGIN
                    WHILE v_order_days_pending > 10 DO
                        SET v_order_status = 'Cancelled';
                        UPDATE orders SET order_status = v_order_status, order_shipping = 0.0 WHERE order_id = order_row.order_id;
                        SET v_order_days_pending = DATEDIFF(DAY, (SELECT order_date FROM orders WHERE order_id = order_row.order_id), CURRENT_TIMESTAMP);
                    END WHILE;
                END L2;
            ELSE
                BEGIN
                    DECLARE v_backlog_order_count INT;
                    SET v_backlog_order_count = (SELECT COUNT(*) FROM orders WHERE order_status = 'Processing' AND DATEDIFF(DAY, order_date, CURRENT_TIMESTAMP) > 5);
                    IF v_backlog_order_count > 0 THEN
                        L3: BEGIN
                            REPEAT
                                INSERT INTO orders (
                                    order_id,
                                    customer_id,
                                    order_date,
                                    order_status,
                                    order_total,
                                    order_tax,
                                    order_discount,
                                    order_shipping
                                ) VALUES (
                                    (SELECT MAX(order_id) + 1 FROM orders),
                                    (SELECT MIN(customer_id) FROM customers),
                                    CURRENT_TIMESTAMP,
                                    'Escalated',
                                    20.0,
                                    2.0,
                                    0.0,
                                    0.0
                                );
                                SET v_backlog_order_count = v_backlog_order_count - 1;
                            UNTIL v_backlog_order_count = 0 END REPEAT;
                        END L3;
                    END IF;
                END;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, WHILE, DML, deep-nesting, exception-handler, backlog-logic
-- EXPECTED: Random status updates, shipping changes, order inserts for backlog maintenance
-- EXECUTES: FOR, IF, REPEAT, WHILE, nested BEGIN...END, exception handler

-- Script 30
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_item_id INT;
    DECLARE v_order_item_status STRING;
    DECLARE v_shipped_item_count INT DEFAULT 0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Order item error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR order_item AS SELECT order_item_id, item_status FROM order_items LIMIT 3 DO
            SET v_order_item_id = order_item.order_item_id;
            SET v_order_item_status = order_item.item_status;
            IF v_order_item_status = 'Shipped' THEN
                SET v_shipped_item_count = v_shipped_item_count + 1;
                L1: BEGIN
                    DECLARE v_item_shipping_delay INT;
                    SET v_item_shipping_delay = (SELECT DATEDIFF(DAY, (SELECT item_shipping_date FROM order_items WHERE order_item_id = v_order_item_id), CURRENT_TIMESTAMP));
                    IF v_item_shipping_delay > 7 THEN
                        REPEAT
                            UPDATE order_items SET item_discount = item_discount + 0.05 WHERE order_item_id = v_order_item_id;
                            SET v_item_shipping_delay = (SELECT DATEDIFF(DAY, (SELECT item_shipping_date FROM order_items WHERE order_item_id = v_order_item_id), CURRENT_TIMESTAMP));
                        UNTIL v_item_shipping_delay < 4 END REPEAT;
                    END IF;
                END L1;
            END IF;
        END FOR;
        BEGIN
            IF v_shipped_item_count < 1 THEN
                L2: BEGIN
                    REPEAT
                        INSERT INTO order_items (
                            order_item_id,
                            order_id,
                            product_id,
                            item_quantity,
                            item_price_per_unit,
                            item_discount,
                            item_tax,
                            item_total,
                            item_status,
                            item_shipping_date
                        ) VALUES (
                            (SELECT MAX(order_item_id) + 1 FROM order_items),
                            (SELECT MIN(order_id) FROM orders),
                            (SELECT MIN(product_id) FROM products),
                            1,
                            10.0,
                            0.0,
                            1.0,
                            11.0,
                            'Shipped',
                            CAST(CURRENT_TIMESTAMP AS DATE)
                        );
                        SET v_shipped_item_count = v_shipped_item_count + 1;
                    UNTIL v_shipped_item_count > 0 END REPEAT;
                END L2;
            END IF;
        END;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, item-logic
-- EXPECTED: Random item discount and status updates, shipped item guarantees
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 31
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_supplier_contact_emails INT DEFAULT 0;
    DECLARE v_active_supplier_count INT DEFAULT 0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Supplier contact error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR supplier AS SELECT supplier_id, supplier_active, supplier_contact_email FROM suppliers LIMIT 3 DO
            IF supplier.supplier_active = true AND supplier.supplier_contact_email IS NOT NULL THEN
                SET v_active_supplier_count = v_active_supplier_count + 1;
                L1: BEGIN
                    REPEAT
                        SET v_supplier_contact_emails = v_supplier_contact_emails + 1;
                        UPDATE suppliers SET supplier_contact_email = 'new_' || supplier.supplier_contact_email WHERE supplier_id = supplier.supplier_id;
                    UNTIL v_supplier_contact_emails > 1 END REPEAT;
                END L1;
                L2: BEGIN
                    DECLARE v_supplier_products INT;
                    SET v_supplier_products = (SELECT COUNT(*) FROM products WHERE supplier_id = supplier.supplier_id);
                    IF v_supplier_products = 0 THEN
                        REPEAT
                            INSERT INTO products VALUES (
                                (SELECT MAX(product_id) + 1 FROM products),
                                'New Supplier Item',
                                'Specialty',
                                35.99,
                                25.99,
                                supplier.supplier_id,
                                'New Contact',
                                'New Role',
                                '123 New St',
                                3,
                                'Red',
                                'Small',
                                0.7,
                                'New supplier item',
                                4.0,
                                0,
                                CURRENT_TIMESTAMP,
                                CURRENT_TIMESTAMP,
                                true,
                                0.0,
                                0.19
                            );
                        UNTIL (SELECT COUNT(*) FROM products WHERE supplier_id = supplier.supplier_id) > 0 END REPEAT;
                    END IF;
                END L2;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, supplier-contact-logic
-- EXPECTED: Random supplier contact and product updates, ensures active supplier content
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 32
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_discount_avg DOUBLE;
    DECLARE v_max_discount_products INT DEFAULT 2;
    DECLARE v_discount_changed BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Product price error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_discount_category STRING DEFAULT 'GENERAL';
        FOR category AS SELECT DISTINCT product_category FROM products LIMIT 2 DO
            SET v_product_discount_avg = (SELECT AVG(product_discount) FROM products WHERE product_category = category.product_category);
            IF v_product_discount_avg > 0.1 THEN
                L1: BEGIN
                    REPEAT
                        SET v_product_discount_avg = v_product_discount_avg - 0.01;
                        UPDATE products SET product_discount = product_discount - 0.01 WHERE product_category = category.product_category;
                        SET v_discount_changed = true;
                    UNTIL v_product_discount_avg <= 0.1 END REPEAT;
                END L1;
            END IF;
            IF v_discount_changed = false THEN
                L2: BEGIN
                    REPEAT
                        INSERT INTO products VALUES (
                            (SELECT COALESCE(MAX(product_id), 0) + 1 FROM products),
                            v_discount_category,
                            'Discount',
                            49.99,
                            39.99,
                            (SELECT MIN(supplier_id) FROM suppliers),
                            'Discounter',
                            'Discount Role',
                            '123 Discount St',
                            5,
                            'Blue',
                            'Large',
                            1.2,
                            'Maximum discount',
                            0.0,
                            0,
                            CURRENT_TIMESTAMP,
                            CURRENT_TIMESTAMP,
                            true,
                            0.9,
                            0.19
                        );
                        SET v_discount_category = v_discount_category || '_EXTRA';
                        SET v_max_discount_products = v_max_discount_products - 1;
                    UNTIL v_max_discount_products < 0 END REPEAT;
                END L2;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, price-discount
-- EXPECTED: Random discount changes and product inserts, category balancing
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 33
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_employee_gender STRING;
    DECLARE v_gender_count INT DEFAULT 0;
    DECLARE v_min_gender_emp INT DEFAULT 1;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Employee diversity error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR gender AS SELECT DISTINCT employee_gender FROM employees LIMIT 2 DO
            SET v_employee_gender = gender.employee_gender;
            SET v_gender_count = (SELECT COUNT(*) FROM employees WHERE employee_gender = v_employee_gender);
            IF v_gender_count < v_min_gender_emp THEN
                L1: BEGIN
                    REPEAT
                        INSERT INTO employees VALUES (
                            (SELECT COALESCE(MAX(employee_id), 0) + 1 FROM employees),
                            v_employee_gender || ' Equality',
                            'diverse@' || LOWER(v_employee_gender) || '.com',
                            '555-0000',
                            'Balancer',
                            'HR',
                            CURRENT_TIMESTAMP,
                            NULL,
                            40000.0,
                            0.0,
                            NULL,
                            '123 Balancer St',
                            NULL,
                            'Balancer City',
                            'CA',
                            '00000',
                            'USA',
                            CURRENT_TIMESTAMP,
                            v_employee_gender,
                            'Single',
                            0
                        );
                        SET v_gender_count = v_gender_count + 1;
                    UNTIL v_gender_count >= v_min_gender_emp END REPEAT;
                END L1;
            END IF;
        END FOR;
    END L0;
    L2: BEGIN
        DECLARE v_fill_role STRING;
        SET v_fill_role = 'Filler';
        REPEAT
            INSERT INTO employees VALUES (
                (SELECT COALESCE(MAX(employee_id), 0) + 1 FROM employees),
                v_fill_role,
                'fill@company.com',
                '555-0000',
                v_fill_role,
                v_fill_role,
                CURRENT_TIMESTAMP,
                NULL,
                30000.0,
                0.0,
                NULL,
                '123 Filler St',
                NULL,
                v_fill_role || 'ville',
                'CA',
                '00000',
                'USA',
                CURRENT_TIMESTAMP,
                'Female',
                'Married',
                1
            );
            SET v_fill_role = v_fill_role || '_R';
        UNTIL (SELECT COUNT(*) FROM employees WHERE employee_position = 'Filler') > 1 END REPEAT;
    END L2;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, diversity-logic
-- EXPECTED: Random employee inserts, gender diversity, position filling
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 34
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_channel STRING;
    DECLARE v_channel_orders INT DEFAULT 0;
    DECLARE v_channel_returns INT DEFAULT 0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Channel analysis error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR channel AS SELECT DISTINCT order_channel FROM orders LIMIT 3 DO
            SET v_order_channel = channel.order_channel;
            SET v_channel_orders = (SELECT COUNT(*) FROM orders WHERE order_channel = v_order_channel);
            SET v_channel_returns = (SELECT COUNT(*) FROM orders o JOIN order_items oi ON o.order_id = oi.order_id WHERE o.order_channel = v_order_channel AND oi.item_status = 'Returned');
            IF v_channel_orders > 0 THEN
                L1: BEGIN
                    IF v_channel_returns > 0 THEN
                        REPEAT
                            UPDATE orders
                            SET order_status = 'Processing',
                                order_tax = order_tax * 0.9
                            WHERE order_id IN (
                                SELECT o.order_id
                                FROM orders o
                                JOIN order_items oi ON o.order_id = oi.order_id
                                WHERE o.order_channel = v_order_channel
                                AND oi.item_status = 'Returned'
                            );
                            SET v_channel_returns = v_channel_returns - 1;
                        UNTIL v_channel_returns = 0 END REPEAT;
                    END IF;
                    BEGIN
                        DECLARE v_low_revenue_orders INT;
                        SET v_low_revenue_orders = (SELECT COUNT(*) FROM orders WHERE order_channel = v_order_channel AND order_total < 25);
                        IF v_low_revenue_orders > 1 THEN
                            L2: BEGIN
                                REPEAT
                                    INSERT INTO order_items (
                                        order_item_id,
                                        order_id,
                                        product_id,
                                        item_quantity,
                                        item_price_per_unit,
                                        item_discount,
                                        item_tax,
                                        item_total
                                    ) VALUES (
                                        (SELECT MAX(order_item_id) + 1 FROM order_items),
                                        (SELECT MIN(order_id) FROM orders WHERE order_channel = v_order_channel AND order_total < 25),
                                        (SELECT MIN(product_id) FROM products),
                                        1,
                                        10.0,
                                        0.0,
                                        1.0,
                                        11.0
                                    );
                                    SET v_low_revenue_orders = v_low_revenue_orders - 1;
                                UNTIL v_low_revenue_orders < 1 END REPEAT;
                            END L2;
                        END IF;
                    END;
                END L1;
            ELSE
                L3: BEGIN
                    REPEAT
                        INSERT INTO orders (
                            order_id,
                            customer_id,
                            order_date,
                            order_status,
                            order_total,
                            order_tax,
                            order_discount,
                            order_shipping,
                            order_channel
                        ) VALUES (
                            (SELECT MAX(order_id) + 1 FROM orders),
                            (SELECT MIN(customer_id) FROM customers),
                            CURRENT_TIMESTAMP,
                            'Channel Fill',
                            15.0,
                            1.5,
                            0.0,
                            2.0,
                            v_order_channel
                        );
                        SET v_channel_orders = v_channel_orders + 1;
                    UNTIL v_channel_orders > 0 END REPEAT;
                END L3;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, channel-logic
-- EXPECTED: Multi-channel order and item updates, random DML
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 35
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_type STRING;
    DECLARE v_type_order_count INT;
    DECLARE v_type_customer_count INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Customer type error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_orders_per_type INT DEFAULT 2;
FOR customer_type AS SELECT DISTINCT customer_type FROM customers LIMIT 3 DO
                     SET v_customer_type = customer_type.customer_type;
SET v_type_order_count = (SELECT COUNT(*) FROM orders o JOIN customers c ON o.customer_id = c.customer_id WHERE c.customer_type = v_customer_type);
            SET v_type_customer_count = (SELECT COUNT(*) FROM customers WHERE customer_type = v_customer_type);
            IF v_type_order_count < v_min_orders_per_type AND v_type_customer_count > 0 THEN
                L1: BEGIN
                    REPEAT
                        INSERT INTO orders (
                            order_id,
                            customer_id,
                            order_date,
                            order_status,
                            order_total,
                            order_tax,
                            order_discount,
                            order_shipping
                        ) VALUES (
                            (SELECT MAX(order_id) + 1 FROM orders),
                            (SELECT MIN(customer_id) FROM customers WHERE customer_type = v_customer_type),
                            CURRENT_TIMESTAMP,
                            'Type Fill',
                            18.0,
                            1.8,
                            0.0,
                            2.2
                        );
                        SET v_type_order_count = v_type_order_count + 1;
                    UNTIL v_type_order_count >= v_min_orders_per_type OR v_type_customer_count = 0 END REPEAT;
END L1;
            ELSEIF v_type_customer_count = 0 THEN
                L2: BEGIN
                    REPEAT
                        INSERT INTO customers VALUES (
                            (SELECT MAX(customer_id) + 1 FROM customers),
                            'Type ' || v_customer_type,
                            'type_' || LOWER(v_customer_type) || '@example.com',
                            '555-0000',
                            '123 Type St',
                            'Type City',
                            'CA',
                            '00000',
                            'USA',
                            v_customer_type,
                            CURRENT_TIMESTAMP,
                            0.0,
                            NULL,
                            NULL,
                            NULL,
                            true,
                            true,
                            0.0,
                            NULL,
                            NULL
                        );
                        SET v_type_customer_count = v_type_customer_count + 1;
                    UNTIL v_type_customer_count > 0 END REPEAT;
END L2;
END IF;
END FOR;
END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, customer-type-logic
-- EXPECTED: Order and customer inserts, ensures coverage by customer type
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 36
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_zip_code STRING;
    DECLARE v_zip_orders INT DEFAULT 0;
    DECLARE v_zip_customers INT DEFAULT 0;
    DECLARE v_zip_customer_total DOUBLE DEFAULT 0.0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Zip code analysis error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_zip_orders INT DEFAULT 3;
        FOR zip AS SELECT DISTINCT customer_zip FROM customers LIMIT 2 DO
            SET v_zip_code = zip.customer_zip;
            SET v_zip_orders = (SELECT COUNT(*) FROM orders o JOIN customers c ON o.customer_id = c.customer_id WHERE c.customer_zip = v_zip_code);
            SET v_zip_customers = (SELECT COUNT(*) FROM customers WHERE customer_zip = v_zip_code);
            IF v_zip_orders < v_min_zip_orders AND v_zip_customers > 0 THEN
                L1: BEGIN
                    REPEAT
                        INSERT INTO orders (
                            order_id,
                            customer_id,
                            order_date,
                            order_status,
                            order_total,
                            order_tax,
                            order_discount,
                            order_shipping
                        ) VALUES (
                            (SELECT MAX(order_id) + 1 FROM orders),
                            (SELECT MIN(customer_id) FROM customers WHERE customer_zip = v_zip_code),
                            CURRENT_TIMESTAMP,
                            'Zip Fill',
                            20.0,
                            2.0,
                            0.0,
                            3.0
                        );
                        SET v_zip_orders = v_zip_orders + 1;
                    UNTIL v_zip_orders >= v_min_zip_orders END REPEAT;
                END L1;
            ELSEIF v_zip_customers = 0 THEN
                L2: BEGIN
                    REPEAT
                        INSERT INTO customers VALUES (
                            (SELECT MAX(customer_id) + 1 FROM customers),
                            'Zip Filler ' || v_zip_code,
                            'zip_' || v_zip_code || '@example.com',
                            '555-0000',
                            '123 Zip St',
                            'Zip City',
                            'CA',
                            v_zip_code,
                            'USA',
                            'Retail',
                            CURRENT_TIMESTAMP,
                            0.0,
                            NULL,
                            NULL,
                            NULL,
                            true,
                            true,
                            0.0,
                            NULL,
                            NULL
                        );
                        SET v_zip_customers = v_zip_customers + 1;
                    UNTIL v_zip_customers > 0 END REPEAT;
                END L2;
            END IF;
            SET v_zip_customer_total = (SELECT SUM(customer_total_spent) FROM customers WHERE customer_zip = v_zip_code);
            IF v_zip_customer_total > 200.0 AND v_zip_customers > 0 THEN
                L3: BEGIN
                    REPEAT
                        UPDATE customers
                        SET customer_discount_percentage = customer_discount_percentage + 0.05
                        WHERE customer_zip = v_zip_code
                        AND customer_discount_percentage < 0.2;
                        SET v_zip_customer_total = (SELECT SUM(customer_total_spent) FROM customers WHERE customer_zip = v_zip_code);
                    UNTIL v_zip_customer_total <= 200.0 OR v_zip_customers = 0 END REPEAT;
                END L3;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, zip-code-logic
-- EXPECTED: Order and customer inserts, zip-level discount logic, balancing
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 37
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_shipping_region STRING;
    DECLARE v_region_shipping_total DOUBLE;
    DECLARE v_region_discount_offered BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Shipping region error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_max_discount_region_order DOUBLE DEFAULT 100.0;
        FOR shipping_region AS SELECT DISTINCT order_shipping_region FROM orders LIMIT 2 DO
            SET v_order_shipping_region = shipping_region.order_shipping_region;
            SET v_region_shipping_total = (SELECT SUM(order_shipping) FROM orders WHERE order_shipping_region = v_order_shipping_region);
            IF v_region_shipping_total > 20 AND v_region_shipping_total < 50 THEN
                L1: BEGIN
                    REPEAT
                        SET v_region_discount_offered = true;
                        UPDATE orders SET order_shipping = order_shipping - 0.5 WHERE order_shipping_region = v_order_shipping_region;
                        SET v_region_shipping_total = (SELECT SUM(order_shipping) FROM orders WHERE order_shipping_region = v_order_shipping_region);
                    UNTIL v_region_shipping_total <= 15 OR v_region_discount_offered = false END REPEAT;
                END L1;
                IF v_region_discount_offered = true THEN
                    BEGIN
                        DECLARE v_low_region_products INT;
                        SET v_low_region_products = (SELECT COUNT(*) FROM order_items oi JOIN orders o ON oi.order_id = o.order_id JOIN products p ON oi.product_id = p.product_id WHERE o.order_shipping_region = v_order_shipping_region AND p.product_discount > 0.1);
                        IF v_low_region_products < 1 THEN
                            REPEAT
                                INSERT INTO products VALUES (
                                    (SELECT MAX(product_id) + 1 FROM products),
                                    'Region Special',
                                    v_order_shipping_region,
                                    44.99,
                                    29.99,
                                    (SELECT MIN(supplier_id) FROM suppliers),
                                    'Region Contact',
                                    'Region Role',
                                    '123 Region St',
                                    3,
                                    'Blue',
                                    'Medium',
                                    0.9,
                                    'Region special',
                                    4.0,
                                    0,
                                    CURRENT_TIMESTAMP,
                                    CURRENT_TIMESTAMP,
                                    true,
                                    0.15,
                                    0.19
                                );
                            UNTIL (SELECT COUNT(*) FROM order_items oi JOIN orders o ON oi.order_id = o.order_id JOIN products p ON oi.product_id = p.product_id WHERE o.order_shipping_region = v_order_shipping_region AND p.product_discount > 0.1) > 0 END REPEAT;
                        END IF;
                    END;
                END IF;
            ELSE
                L2: BEGIN
                    REPEAT
                        INSERT INTO orders (
                            order_id,
                            customer_id,
                            order_date,
                            order_status,
                            order_total,
                            order_tax,
                            order_discount,
                            order_shipping,
                            order_shipping_region
                        ) VALUES (
                            (SELECT MAX(order_id) + 1 FROM orders),
                            (SELECT MIN(customer_id) FROM customers),
                            CURRENT_TIMESTAMP,
                            'Region Fill',
                            22.0,
                            2.2,
                            0.0,
                            5.0,
                            v_order_shipping_region
                        );
                        SET v_region_shipping_total = (SELECT SUM(order_shipping) FROM orders WHERE order_shipping_region = v_order_shipping_region);
                    UNTIL v_region_shipping_total > 10 END REPEAT;
                END L2;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, shipping-region-logic
-- EXPECTED: Shipping and product updates, region-specific logic, DML
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 38
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_email_domain STRING;
    DECLARE v_domain_customer_count INT;
    DECLARE v_domain_order_count INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Email domain error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_orders_per_domain INT DEFAULT 1;
        FOR customer AS SELECT SUBSTRING(customer_email, POSITION('@' IN customer_email)) AS domain, customer_id FROM customers LIMIT 3 DO
            SET v_customer_email_domain = customer.domain;
            SET v_domain_customer_count = (SELECT COUNT(*) FROM customers WHERE customer_email LIKE '%' || v_customer_email_domain);
            SET v_domain_order_count = (SELECT COUNT(*) FROM orders WHERE customer_id IN (SELECT customer_id FROM customers WHERE customer_email LIKE '%' || v_customer_email_domain));
            IF v_domain_order_count < v_min_orders_per_domain AND v_domain_customer_count > 0 THEN
                L1: BEGIN
                    REPEAT
                        INSERT INTO orders (
                            order_id,
                            customer_id,
                            order_date,
                            order_status,
                            order_total,
                            order_tax,
                            order_discount,
                            order_shipping
                        ) VALUES (
                            (SELECT MAX(order_id) + 1 FROM orders),
                            customer.customer_id,
                            CURRENT_TIMESTAMP,
                            'Domain Fill',
                            15.0,
                            1.5,
                            0.0,
                            2.0
                        );
                        SET v_domain_order_count = v_domain_order_count + 1;
                    UNTIL v_domain_order_count >= v_min_orders_per_domain END REPEAT;
                END L1;
            END IF;
        END FOR;
    END L0;
    BEGIN
        DECLARE v_empty_domain_count INT;
        SET v_empty_domain_count = (SELECT COUNT(*) FROM customers WHERE customer_email IS NULL OR customer_email = '');
        IF v_empty_domain_count > 0 THEN
            L2: BEGIN
                REPEAT
                    INSERT INTO order_items (
                        order_item_id,
                        order_id,
                        product_id,
                        item_quantity,
                        item_price_per_unit,
                        item_discount,
                        item_tax,
                        item_total
                    ) VALUES (
                        (SELECT MAX(order_item_id) + 1 FROM order_items),
                        (SELECT MIN(order_id) FROM orders),
                        (SELECT MIN(product_id) FROM products),
                        1,
                        10.0,
                        0.0,
                        1.0,
                        11.0
                    );
                    SET v_empty_domain_count = v_empty_domain_count - 1;
                UNTIL v_empty_domain_count = 0 END REPEAT;
            END L2;
        END IF;
    END;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, email-domain-logic
-- EXPECTED: Order and item inserts, email domain logic, DML
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 39
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_department_emp_count INT;
    DECLARE v_department_avg_yearly_salary DOUBLE;
    DECLARE v_salary_change_needed BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Department salary error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_dept_size INT DEFAULT 2;
        FOR department AS SELECT DISTINCT employee_department FROM employees LIMIT 3 DO
            SET v_department_emp_count = (SELECT COUNT(*) FROM employees WHERE employee_department = department.employee_department);
            SET v_department_avg_yearly_salary = (SELECT AVG(employee_salary) FROM employees WHERE employee_department = department.employee_department);
            IF v_department_emp_count > v_min_dept_size THEN
                L1: BEGIN
                    IF v_department_avg_yearly_salary > 60000 THEN
                        L2: BEGIN
                            SET v_salary_change_needed = true;
                            REPEAT
                                UPDATE employees SET employee_salary = employee_salary * 0.97 WHERE employee_department = department.employee_department AND employee_salary > 40000;
                                SET v_department_avg_yearly_salary = (SELECT AVG(employee_salary) FROM employees WHERE employee_department = department.employee_department);
                            UNTIL v_department_avg_yearly_salary < 60000 OR v_salary_change_needed = false END REPEAT;
                        END L2;
                    END IF;
                END L1;
            ELSE
                L3: BEGIN
                    DECLARE v_new_emp_id INT;
                    SET v_new_emp_id = (SELECT MAX(employee_id) + 1 FROM employees);
                    INSERT INTO employees VALUES (
                        v_new_emp_id,
                        'Dept Filler',
                        'dept_' || LOWER(department.employee_department) || '@company.com',
                        '555-0000',
                        'Member',
                        department.employee_department,
                        CURRENT_TIMESTAMP,
                        NULL,
                        45000.0,
                        0.0,
                        NULL,
                        '123 Dept St',
                        NULL,
                        'Dept City',
                        'CA',
                        '00000',
                        'USA',
                        CURRENT_TIMESTAMP,
                        'Female',
                        'Single',
                        0
                    );
                    SET v_department_emp_count = v_department_emp_count + 1;
                END L3;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, department-salary-logic
-- EXPECTED: Random salary and department updates, DML
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 40
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_status STRING;
    DECLARE v_status_count INT;
    DECLARE v_status_order_total DOUBLE;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Order status error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_status_orders INT DEFAULT 2;
        FOR order_stat AS SELECT DISTINCT order_status FROM orders LIMIT 3 DO
            SET v_order_status = order_stat.order_status;
            SET v_status_count = (SELECT COUNT(*) FROM orders WHERE order_status = v_order_status);
            SET v_status_order_total = (SELECT SUM(order_total) FROM orders WHERE order_status = v_order_status);
            IF v_status_count < v_min_status_orders THEN
                L1: BEGIN
                    REPEAT
                        INSERT INTO orders (
                            order_id,
                            customer_id,
                            order_date,
                            order_status,
                            order_total,
                            order_tax,
                            order_discount,
                            order_shipping
                        ) VALUES (
                            (SELECT MAX(order_id) + 1 FROM orders),
                            (SELECT MIN(customer_id) FROM customers),
                            CURRENT_TIMESTAMP,
                            v_order_status,
                            15.0,
                            1.5,
                            0.0,
                            2.0
                        );
                        SET v_status_count = (SELECT COUNT(*) FROM orders WHERE order_status = v_order_status);
                    UNTIL v_status_count >= v_min_status_orders END REPEAT;
                END L1;
            END IF;
            IF v_status_order_total > 200 AND v_status_count > 1 THEN
                L2: BEGIN
                    REPEAT
                        SET v_status_order_total = v_status_order_total - 10.0;
                        UPDATE orders
                        SET order_total = order_total - 10.0
                        WHERE order_status = v_order_status
                        AND order_total > 30.0;
                    UNTIL v_status_order_total <= 200 OR v_status_count < 2 END REPEAT;
                END L2;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, status-order-logic
-- EXPECTED: Random order inserts, status balancing, revenue control
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 41
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_emp_department_size_counts INT;
    DECLARE v_dept_name STRING;
    DECLARE v_dept_average_age INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Department diversity error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_dept_diversity INT DEFAULT 2;
        FOR dept AS SELECT DISTINCT employee_department FROM employees LIMIT 3 DO
            SET v_dept_name = dept.employee_department;
            SET v_emp_department_size_counts = (SELECT COUNT(*) FROM employees WHERE employee_department = v_dept_name);
            IF v_emp_department_size_counts < v_min_dept_diversity THEN
                L1: BEGIN
                    REPEAT
                        INSERT INTO employees VALUES (
                            (SELECT MAX(employee_id) + 1 FROM employees),
                            'Diverse Dept ' || v_dept_name,
                            'diverse_' || LOWER(v_dept_name) || '@company.com',
                            '555-0000',
                            'Balancer',
                            v_dept_name,
                            CURRENT_TIMESTAMP,
                            NULL,
                            35000.0,
                            0.0,
                            NULL,
                            '123 Balance St',
                            NULL,
                            'Balance City',
                            'CA',
                            '00000',
                            'USA',
                            CASE WHEN MOD((SELECT COUNT(*) FROM employees WHERE employee_department = v_dept_name), 2) = 0 THEN 'Male' ELSE 'Female' END,
                            'Single',
                            0
                        );
                        SET v_emp_department_size_counts = v_emp_department_size_counts + 1;
                    UNTIL v_emp_department_size_counts >= v_min_dept_diversity END REPEAT;
                END L1;
            END IF;
            SET v_dept_average_age = (SELECT AVG(DATEDIFF(YEAR, employee_birth_date, CURRENT_TIMESTAMP)) FROM employees WHERE employee_department = v_dept_name);
            IF v_dept_average_age IS NULL THEN
                L2: BEGIN
                    REPEAT
                        INSERT INTO employees VALUES (
                            (SELECT MAX(employee_id) + 1 FROM employees),
                            'Age Dept ' || v_dept_name,
                            'age_' || LOWER(v_dept_name) || '@company.com',
                            '555-0000',
                            'Age',
                            v_dept_name,
                            CURRENT_TIMESTAMP,
                            NULL,
                            30000.0,
                            0.0,
                            NULL,
                            '123 Age St',
                            NULL,
                            'Age City',
                            'CA',
                            '00000',
                            'USA',
                            DATEADD(YEAR, -30, CURRENT_TIMESTAMP),
                            'Female',
                            'Married',
                            1
                        );
                        SET v_dept_average_age = (SELECT AVG(DATEDIFF(YEAR, employee_birth_date, CURRENT_TIMESTAMP)) FROM employees WHERE employee_department = v_dept_name);
                    UNTIL v_dept_average_age IS NOT NULL END REPEAT;
                END L2;
            ELSEIF v_dept_average_age < 30 THEN
                L3: BEGIN
                    REPEAT
                        INSERT INTO employees VALUES (
                            (SELECT MAX(employee_id) + 1 FROM employees),
                            'Mature ' || v_dept_name,
                            'mature_' || LOWER(v_dept_name) || '@company.com',
                            '555-0000',
                            'Senior',
                            v_dept_name,
                            CURRENT_TIMESTAMP,
                            NULL,
                            55000.0,
                            0.0,
                            NULL,
                            '123 Senior St',
                            NULL,
                            'Senior City',
                            'CA',
                            '00000',
                            'USA',
                            DATEADD(YEAR, -40, CURRENT_TIMESTAMP),
                            'Male',
                            'Married',
                            3
                        );
                        SET v_dept_average_age = (SELECT AVG(DATEDIFF(YEAR, employee_birth_date, CURRENT_TIMESTAMP)) FROM employees WHERE employee_department = v_dept_name);
                    UNTIL v_dept_average_age > 30 END REPEAT;
                END L3;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, department-diversity-logic
-- EXPECTED: Employee inserts, age and gender balancing, DML
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 42
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_supplier_type STRING;
    DECLARE v_supplier_product_count INT;
    DECLARE v_products_per_type_min INT DEFAULT 3;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Supplier product error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR supplier AS SELECT supplier_id, supplier_main_products FROM suppliers LIMIT 3 DO
            SET v_supplier_type = supplier.supplier_main_products;
            SET v_supplier_product_count = (SELECT COUNT(*) FROM products WHERE supplier_id = supplier.supplier_id AND product_category = v_supplier_type);
            IF v_supplier_product_count < v_products_per_type_min THEN
                L1: BEGIN
                    REPEAT
                        INSERT INTO products VALUES (
                            (SELECT MAX(product_id) + 1 FROM products),
                            'Type ' || v_supplier_type,
                            v_supplier_type,
                            39.99,
                            29.99,
                            supplier.supplier_id,
                            supplier.supplier_main_products || ' Contact',
                            supplier.supplier_main_products || ' Role',
                            '123 Type St',
                            7,
                            'Mixed',
                            'Standard',
                            0.8,
                            'Type product',
                            3.5,
                            0,
                            CURRENT_TIMESTAMP,
                            CURRENT_TIMESTAMP,
                            true,
                            0.0,
                            0.19
                        );
                        SET v_supplier_product_count = v_supplier_product_count + 1;
                    UNTIL v_supplier_product_count >= v_products_per_type_min END REPEAT;
                END L1;
            END IF;
        END FOR;
    END L0;
    BEGIN
        DECLARE v_no_match_product_count INT;
        SET v_no_match_product_count = (
            SELECT COUNT(*)
            FROM products p
            JOIN suppliers s
                ON p.supplier_id = s.supplier_id
            WHERE p.product_category != s.supplier_main_products
        );
        IF v_no_match_product_count > 0 THEN
            L2: BEGIN
                REPEAT
                    UPDATE products
                    SET product_category = (
                        SELECT supplier_main_products
                        FROM suppliers
                        WHERE supplier_id = products.supplier_id
                        LIMIT 1
                    )
                    WHERE product_id = (
                        SELECT MIN(product_id)
                        FROM products p
                        JOIN suppliers s
                            ON p.ssupplier_id = s.supplier_id
                        WHERE p.product_category != s.supplier_main_products
                        LIMIT 1
                    );
                    SET v_no_match_product_count = (
                        SELECT COUNT(*)
                        FROM products p
                        JOIN suppliers s
                            ON p.supplier_id = s.supplier_id
                        WHERE p.product_category != s.supplier_main_products
                    );
                UNTIL v_no_match_product_count = 0 END REPEAT;
            END L2;
        END IF;
    END;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, supplier-product-logic
-- EXPECTED: Random product inserts, supplier-category matching, DML
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 43
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_zip_orders_total DOUBLE;
    DECLARE v_zip_code STRING;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Customer zip order error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_zip_min_orders_total DOUBLE DEFAULT 100.0;
        FOR zip AS SELECT DISTINCT customer_zip FROM customers LIMIT 3 DO
            SET v_zip_code = zip.customer_zip;
            SET v_customer_zip_orders_total = (
                SELECT SUM(order_total)
                FROM orders
                WHERE customer_id IN (
                    SELECT customer_id
                    FROM customers
                    WHERE customer_zip = v_zip_code
                )
            );
            IF v_customer_zip_orders_total < v_zip_min_orders_total THEN
                L1: BEGIN
                    REPEAT
                        INSERT INTO orders (
                            order_id,
                            customer_id,
                            order_date,
                            order_status,
                            order_total,
                            order_tax,
                            order_discount,
                            order_shipping
                        ) VALUES (
                            (SELECT MAX(order_id) + 1 FROM orders),
                            (SELECT MIN(customer_id) FROM customers WHERE customer_zip = v_zip_code),
                            CURRENT_TIMESTAMP,
                            'Zip Revenue',
                            20.0,
                            2.0,
                            0.0,
                            2.0
                        );
                        SET v_customer_zip_orders_total = (
                            SELECT SUM(order_total)
                            FROM orders
                            WHERE customer_id IN (
                                SELECT customer_id
                                FROM customers
                                WHERE customer_zip = v_zip_code
                            )
                        );
                    UNTIL v_customer_zip_orders_total >= v_zip_min_orders_total END REPEAT;
                END L1;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, zip-revenue-logic
-- EXPECTED: Order inserts, local revenue control, DML
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 44
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_device_type STRING;
    DECLARE v_device_order_count INT DEFAULT 0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Order device error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_device_orders INT DEFAULT 1;
        FOR device AS SELECT DISTINCT order_device FROM orders LIMIT 3 DO
            SET v_order_device_type = device.order_device;
            SET v_device_order_count = (SELECT COUNT(*) FROM orders WHERE order_device = v_order_device_type);
            IF v_device_order_count < v_min_device_orders THEN
                L1: BEGIN
                    REPEAT
                        INSERT INTO orders (
                            order_id,
                            customer_id,
                            order_date,
                            order_status,
                            order_total,
                            order_tax,
                            order_discount,
                            order_shipping,
                            order_device
                        ) VALUES (
                            (SELECT MAX(order_id) + 1 FROM orders),
                            (SELECT MIN(customer_id) FROM customers),
                            CURRENT_TIMESTAMP,
                            'Device Fill',
                            15.0,
                            1.5,
                            0.0,
                            2.0,
                            v_order_device_type
                        );
                        SET v_device_order_count = v_device_order_count + 1;
                    UNTIL v_device_order_count >= v_min_device_orders END REPEAT;
                END L1;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, device-logic
-- EXPECTED: Order inserts, device diversity, DML
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 45
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_cost DOUBLE;
    DECLARE v_product_discount DOUBLE;
    DECLARE v_low_margin_flag BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Product margin error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_products_margin_check INT DEFAULT 2;
        FOR product AS SELECT product_id, product_cost, product_discount FROM products LIMIT 3 DO
            SET v_product_cost = product.product_cost;
            SET v_product_discount = product.product_discount;
            L1: BEGIN
                DECLARE v_sale_price DOUBLE;
                SET v_sale_price = (SELECT product_price FROM products WHERE product_id = product.product_id);
                IF v_sale_price - v_product_cost < 10 THEN
                    SET v_low_margin_flag = true;
                    L2: BEGIN
                        REPEAT
                            UPDATE products
                            SET product_price = product_price * 1.05
                            WHERE product_id = product.product_id;
                            SET v_sale_price = (SELECT product_price FROM products WHERE product_id = product.product_id);
                        UNTIL v_sale_price - v_product_cost >= 10 OR v_low_margin_flag = false END REPEAT;
                    END L2;
                END IF;
                IF v_product_discount > 0.1 AND v_low_margin_flag = true THEN
                    REPEAT
                        UPDATE products
                        SET product_discount = product_discount - 0.02
                        WHERE product_id = product.product_id;
                        SET v_product_discount = (SELECT product_discount FROM products WHERE product_id = product.product_id);
                    UNTIL v_product_discount <= 0.1 END REPEAT;
                END IF;
            END L1;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, margin-logic
-- EXPECTED: Random price and discount updates, margin control, DML
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 46
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_item_count INT;
    DECLARE v_order_id INT;
    DECLARE v_order_total DOUBLE;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Order item counting error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR order_id_row AS SELECT order_id FROM orders LIMIT 3 DO
            SET v_order_id = order_id_row.order_id;
            SET v_order_total = (SELECT order_total FROM orders WHERE order_id = v_order_id);
            SET v_order_item_count = (SELECT COUNT(*) FROM order_items WHERE order_id = v_order_id);
            IF v_order_item_count = 0 THEN
                L1: BEGIN
                    REPEAT
                        INSERT INTO order_items (
                            order_item_id,
                            order_id,
                            product_id,
                            item_quantity,
                            item_price_per_unit,
                            item_discount,
                            item_tax,
                            item_total
                        ) VALUES (
                            (SELECT MAX(order_item_id) + 1 FROM order_items),
                            v_order_id,
                            (SELECT MIN(product_id) FROM products),
                            1,
                            10.0,
                            0.0,
                            1.0,
                            11.0
                        );
                        SET v_order_item_count = (SELECT COUNT(*) FROM order_items WHERE order_id = v_order_id);
                    UNTIL v_order_item_count > 0 END REPEAT;
                END L1;
            END IF;
            IF v_order_total < 10 AND v_order_item_count > 0 THEN
                L2: BEGIN
                    REPEAT
                        UPDATE orders
                        SET order_total = order_total + 5.0
                        WHERE order_id = v_order_id;
                        SET v_order_total = (SELECT order_total FROM orders WHERE order_id = v_order_id);
                    UNTIL v_order_total >= 10 END REPEAT;
                END L2;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, order-item-logic
-- EXPECTED: Order item and total updates, order content enforcement, DML
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 47
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_payment_balance DOUBLE;
    DECLARE v_payment_status STRING;
    DECLARE v_unpaid_orders INT DEFAULT 0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Order payment error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_payment_update_needed BOOLEAN DEFAULT false;
        FOR payment_row AS SELECT order_id, order_total, order_tax, order_payment_status FROM orders LIMIT 3 DO
            SET v_order_payment_balance = payment_row.order_total + payment_row.order_tax;
            SET v_payment_status = payment_row.order_payment_status;
            IF v_payment_status = 'Pending' AND v_order_payment_balance > 20 THEN
                L1: BEGIN
                    SET v_payment_update_needed = true;
                    REPEAT
                        UPDATE orders
                        SET order_payment_status = 'Paid'
                        WHERE order_id = payment_row.order_id;
                        SET v_payment_status = (SELECT order_payment_status FROM orders WHERE order_id = payment_row.order_id);
                    UNTIL v_payment_status != 'Pending' OR v_payment_update_needed = false END REPEAT;
                END L1;
            END IF;
        END FOR;
    END L0;
    BEGIN
        SET v_unpaid_orders = (SELECT COUNT(*) FROM orders WHERE order_payment_status = 'Pending' AND (order_total + order_tax) > 20);
        IF v_unpaid_orders > 0 THEN
            L2: BEGIN
                REPEAT
                    INSERT INTO customers (
                        customer_id,
                        customer_name,
                        customer_email,
                        customer_phone,
                        customer_address_1,
                        customer_city,
                        customer_state,
                        customer_zip,
                        customer_country,
                        customer_type,
                        customer_total_spent,
                        customer_active_status
                    ) VALUES (
                        (SELECT MAX(customer_id) + 1 FROM customers),
                        'Pending Payee',
                        'pending@payment.com',
                        '555-0000',
                        '123 Pending St',
                        'Paymentville',
                        'CA',
                        '00000',
                        'USA',
                        'Wholesale',
                        0.0,
                        true
                    );
                    INSERT INTO orders (
                        order_id,
                        customer_id,
                        order_date,
                        order_status,
                        order_total,
                        order_tax,
                        order_discount,
                        order_shipping,
                        order_payment_status
                    ) VALUES (
                        (SELECT MAX(order_id) + 1 FROM orders),
                        (SELECT MAX(customer_id) FROM customers),
                        CURRENT_TIMESTAMP,
                        'Generated',
                        25.0,
                        2.5,
                        0.0,
                        3.0,
                        'Pending'
                    );
                    SET v_unpaid_orders = v_unpaid_orders - 1;
                UNTIL v_unpaid_orders = 0 END REPEAT;
            END L2;
        END IF;
    END;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, payment-logic
-- EXPECTED: Payment status updates, customer/order inserts, unpaid order control
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 48
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_supplier_active BOOLEAN;
    DECLARE v_suppliers_per_region_target INT DEFAULT 2;
    DECLARE v_region_suppliers_current INT;
    DECLARE v_region_to_balance STRING;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Supplier region error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR region AS SELECT DISTINCT supplier_region FROM suppliers LIMIT 3 DO
            SET v_region_to_balance = region.supplier_region;
            SET v_region_suppliers_current = (SELECT COUNT(*) FROM suppliers WHERE supplier_active = true AND supplier_region = v_region_to_balance);
            IF v_region_suppliers_current < v_suppliers_per_region_target THEN
                L1: BEGIN
                    REPEAT
                        INSERT INTO suppliers VALUES (
                            (SELECT MAX(supplier_id) + 1 FROM suppliers),
                            'Regional Filler',
                            'Regional Manager',
                            'Contact Role',
                            'region_' || v_region_to_balance || '@example.com',
                            '555-0001',
                            '123 Regional St',
                            'Regional City',
                            'CA',
                            '00000',
                            'USA',
                            v_region_to_balance,
                            'Diverse',
                            'http://regional.com',
                            CURRENT_TIMESTAMP,
                            NULL,
                            NULL,
                            4.0,
                            true
                        );
                        SET v_region_suppliers_current = (SELECT COUNT(*) FROM suppliers WHERE supplier_active = true AND supplier_region = v_region_to_balance);
                    UNTIL v_region_suppliers_current >= v_suppliers_per_region_target END REPEAT;
                END L1;
            END IF;
        END FOR;
    END L0;
    BEGIN
        DECLARE v_no_region_suppliers_count INT;
        SET v_no_region_suppliers_count = (SELECT COUNT(DISTINCT supplier_region) FROM suppliers);
        IF v_no_region_suppliers_count < 5 THEN
            L2: BEGIN
                REPEAT
                    INSERT INTO suppliers VALUES (
                        (SELECT MAX(supplier_id) + 1 FROM suppliers),
                        'New Region Corp',
                        'Region Manager',
                        'Contact',
                        'newregion@example.com',
                        '555-0002',
                        '123 Main St',
                        'New City',
                        'CA',
                        '00001',
                        'USA',
                        CONCAT('Region_', v_no_region_suppliers_count + 1),
                        'General',
                        'http://newregion.com',
                        CURRENT_TIMESTAMP,
                        NULL,
                        NULL,
                        4.5,
                        true
                    );
                    SET v_no_region_suppliers_count = (SELECT COUNT(DISTINCT supplier_region) FROM suppliers);
                UNTIL v_no_region_suppliers_count >= 5 END REPEAT;
            END L2;
        END IF;
    END;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, region-logic
-- EXPECTED: Supplier inserts, region balancing, diversity control
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 49
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_color_diversity INT;
    DECLARE v_color_product_count INT;
    DECLARE v_color_to_boost STRING;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Color diversity error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_color_products INT DEFAULT 2;
        FOR color AS SELECT DISTINCT product_color FROM products LIMIT 3 DO
            SET v_color_to_boost = color.product_color;
            SET v_color_product_count = (SELECT COUNT(*) FROM products WHERE product_color = v_color_to_boost);
            IF v_color_product_count < v_min_color_products THEN
                L1: BEGIN
                    REPEAT
                        INSERT INTO products VALUES (
                            (SELECT MAX(product_id) + 1 FROM products),
                            'Boost ' || v_color_to_boost,
                            'Diverse',
                            29.99,
                            19.99,
                            (SELECT MIN(supplier_id) FROM suppliers),
                            'Color Contact',
                            'Color Role',
                            '123 Color St',
                            5,
                            v_color_to_boost,
                            'Standard',
                            0.7,
                            'Diverse color',
                            4.0,
                            0,
                            CURRENT_TIMESTAMP,
                            CURRENT_TIMESTAMP,
                            true,
                            0.05,
                            0.19
                        );
                        SET v_color_product_count = (SELECT COUNT(*) FROM products WHERE product_color = v_color_to_boost);
                    UNTIL v_color_product_count >= v_min_color_products END REPEAT;
                END L1;
            END IF;
        END FOR;
    END L0;
    BEGIN
        SET v_product_color_diversity = (SELECT COUNT(DISTINCT product_color) FROM products);
        IF v_product_color_diversity < 6 THEN
            L2: BEGIN
                REPEAT
                    INSERT INTO products VALUES (
                        (SELECT MAX(product_id) + 1 FROM products),
                        'Colorful New',
                        'General',
                        24.99,
                        14.99,
                        (SELECT MIN(supplier_id) FROM suppliers),
                        'Diverse Contact',
                        'Diverse Role',
                        '123 Diverse St',
                        8,
                        CASE v_product_color_diversity
                            WHEN 0 THEN 'Red'
                            WHEN 1 THEN 'Green'
                            WHEN 2 THEN 'Blue'
                            WHEN 3 THEN 'Yellow'
                            WHEN 4 THEN 'Purple'
                            ELSE 'Orange'
                        END,
                        'Small',
                        0.5,
                        'Expanding diversity',
                        3.0,
                        0,
                        CURRENT_TIMESTAMP,
                        CURRENT_TIMESTAMP,
                        true,
                        0.0,
                        0.19
                    );
                    SET v_product_color_diversity = (SELECT COUNT(DISTINCT product_color) FROM products);
                UNTIL v_product_color_diversity >= 6 END REPEAT;
            END L2;
        END IF;
    END;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, color-diversity
-- EXPECTED: Product color diversity, inserts, DML
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 50
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_origin_diversity INT;
    DECLARE v_origin_supplier_pairs INT;
    DECLARE v_origin_to_activate STRING;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Product origin error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_origin_types INT DEFAULT 3;
        SET v_product_origin_diversity = (SELECT COUNT(DISTINCT supplier_country) FROM products p JOIN suppliers s ON p.supplier_id = s.supplier_id);
        IF v_product_origin_diversity < v_min_origin_types THEN
            L1: BEGIN
                REPEAT
                    SET v_origin_to_activate = CASE v_product_origin_diversity
                                                WHEN 0 THEN 'Canada'
                                                WHEN 1 THEN 'Mexico'
                                                WHEN 2 THEN 'UK'
                                                ELSE 'Australia'
                                              END;
                    INSERT INTO suppliers VALUES (
                        (SELECT MAX(supplier_id) + 1 FROM suppliers),
                        v_origin_to_activate || ' Exports',
                        'Export Manager',
                        'Contact',
                        'exports@' || LOWER(v_origin_to_activate) || '.com',
                        '555-0003',
                        '123 Export St',
                        v_origin_to_activate || ' City',
                        v_origin_to_activate,
                        '00000',
                        v_origin_to_activate,
                        v_origin_to_activate,
                        'General',
                        'http://exports.com',
                        CURRENT_TIMESTAMP,
                        NULL,
                        NULL,
                        4.0,
                        true
                    );
                    INSERT INTO products VALUES (
                        (SELECT MAX(product_id) + 1 FROM products),
                        v_origin_to_activate || ' Import',
                        'Specialty',
                        34.99,
                        24.99,
                        (SELECT MAX(supplier_id) FROM suppliers WHERE supplier_country = v_origin_to_activate),
                        'Importer',
                        'Import Role',
                        '123 Import St',
                        4,
                        'Any',
                        'Medium',
                        0.6,
                        'Imported',
                        4.5,
                        0,
                        CURRENT_TIMESTAMP,
                        CURRENT_TIMESTAMP,
                        true,
                        0.0,
                        0.19
                    );
                    SET v_product_origin_diversity = (SELECT COUNT(DISTINCT supplier_country) FROM products p JOIN suppliers s ON p.supplier_id = s.supplier_id);
                UNTIL v_product_origin_diversity >= v_min_origin_types END REPEAT;
            END L1;
        END IF;
        SET v_origin_supplier_pairs = (SELECT COUNT(*) FROM products p JOIN suppliers s ON p.supplier_id = s.supplier_id WHERE p.product_category = 'Specialty');
        IF v_origin_supplier_pairs < 2 THEN
            L2: BEGIN
                REPEAT
                    INSERT INTO products VALUES (
                        (SELECT MAX(product_id) + 1 FROM products),
                        'Diverse Import',
                        'Specialty',
                        39.99,
                        29.99,
                        (SELECT MIN(supplier_id) FROM suppliers WHERE supplier_country != 'USA'),
                        'Diverse Contact',
                        'Diverse Role',
                        '123 Diverse St',
                        5,
                        'Multi',
                        'Large',
                        0.8,
                        'Diverse',
                        4.0,
                        0,
                        CURRENT_TIMESTAMP,
                        CURRENT_TIMESTAMP,
                        true,
                        0.0,
                        0.19
                    );
                    SET v_origin_supplier_pairs = (SELECT COUNT(*) FROM products p JOIN suppliers s ON p.supplier_id = s.supplier_id WHERE p.product_category = 'Specialty');
                UNTIL v_origin_supplier_pairs >= 2 END REPEAT;
            END L2;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, REPEAT, DML, deep-nesting, exception-handler, origin-diversity
-- EXPECTED: Product and supplier origin diversity, international coverage, DML
-- EXECUTES: FOR, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 51
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_city STRING;
    DECLARE v_city_order_count INT;
    DECLARE v_city_supplier_count INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('City order diversity error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_orders_per_city INT DEFAULT 1;
        DECLARE v_min_suppliers_per_city INT DEFAULT 1;
        FOR city AS SELECT DISTINCT customer_city FROM customers LIMIT 3 DO
            SET v_customer_city = city.customer_city;
            SET v_city_order_count = (SELECT COUNT(*) FROM orders WHERE customer_id IN (SELECT customer_id FROM customers WHERE customer_city = v_customer_city));
            IF v_city_order_count = 0 THEN
                L1: BEGIN
                    INSERT INTO orders (
                        order_id,
                        customer_id,
                        order_date,
                        order_status,
                        order_total,
                        order_tax,
                        order_discount,
                        order_shipping
                    ) VALUES (
                        (SELECT MAX(order_id) + 1 FROM orders),
                        (SELECT MIN(customer_id) FROM customers WHERE customer_city = v_customer_city),
                        CURRENT_TIMESTAMP,
                        'City Fill',
                        25.0,
                        2.5,
                        0.0,
                        2.5
                    );
                    SET v_city_order_count = (SELECT COUNT(*) FROM orders WHERE customer_id IN (SELECT customer_id FROM customers WHERE customer_city = v_customer_city));
                END L1;
                L2: BEGIN
                    SET v_city_supplier_count = (SELECT COUNT(*) FROM suppliers WHERE supplier_city = v_customer_city);
                    IF v_city_supplier_count = 0 THEN
                        INSERT INTO suppliers VALUES (
                            (SELECT MAX(supplier_id) + 1 FROM suppliers),
                            'City ' || v_customer_city,
                            'City Contact',
                            'City Role',
                            'city@example.com',
                            '555-0000',
                            '123 City St',
                            v_customer_city,
                            'CA',
                            '00000',
                            'USA',
                            v_customer_city,
                            'General',
                            'http://city.com',
                            CURRENT_TIMESTAMP,
                            NULL,
                            NULL,
                            4.0,
                            true
                        );
                        SET v_city_supplier_count = v_city_supplier_count + 1;
                    END IF;
                END L2;
            END IF;
            IF EXISTS (SELECT 1 FROM customers WHERE customer_city = v_customer_city) THEN
                L3: BEGIN
                    CASE
                        WHEN v_city_order_count % 2 = 0 THEN
                            UPDATE customers
                            SET customer_active_status = true
                            WHERE customer_city = v_customer_city;
                        ELSE
                            UPDATE customers
                            SET customer_active_status = false
                            WHERE customer_city = v_customer_city
                            AND customer_id = (
                                SELECT MIN(customer_id)
                                FROM customers
                                WHERE customer_city = v_customer_city
                            );
                    END CASE;
                END L3;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, CASE, DML, deep-nesting, exception-handler, city-diversity-logic
-- EXPECTED: Order, supplier, and status updates, city diversity, DML
-- EXECUTES: FOR, IF, CASE, nested BEGIN...END, exception handler

-- Script 52
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_size STRING;
    DECLARE v_size_variety INT;
    DECLARE v_new_sizes_to_add INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Product size error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        SET v_size_variety = (SELECT COUNT(DISTINCT product_size) FROM products);
        SET v_new_sizes_to_add = 5 - v_size_variety;
        IF v_new_sizes_to_add > 0 THEN
            L1: BEGIN
                WHILE v_new_sizes_to_add > 0 DO
                    SET v_product_size = CASE v_new_sizes_to_add
                        WHEN 5 THEN 'XS'
                        WHEN 4 THEN 'S'
                        WHEN 3 THEN 'M'
                        WHEN 2 THEN 'L'
                        WHEN 1 THEN 'XL'
                    END;
                    INSERT INTO products VALUES (
                        (SELECT MAX(product_id) + 1 FROM products),
                        'Size Booster ' || v_product_size,
                        'Diverse',
                        24.99,
                        14.99,
                        (SELECT MIN(supplier_id) FROM suppliers),
                        'Diverse Contact',
                        'Size Role',
                        '123 Size St',
                        6,
                        'Blue',
                        v_product_size,
                        0.6,
                        'Size variety',
                        3.5,
                        0,
                        CURRENT_TIMESTAMP,
                        CURRENT_TIMESTAMP,
                        true,
                        0.05,
                        0.19
                    );
                    SET v_new_sizes_to_add = v_new_sizes_to_add - 1;
                    SET v_size_variety = (SELECT COUNT(DISTINCT product_size) FROM products);
                END WHILE;
            END L1;
        END IF;
        L2: BEGIN
            FOR size AS SELECT DISTINCT product_size FROM products LIMIT 1 DO
                SET v_product_size = size.product_size;
                IF (SELECT COUNT(*) FROM products WHERE product_size = v_product_size) < 2 THEN
                    BEGIN
                        SET v_new_sizes_to_add = 2 - (SELECT COUNT(*) FROM products WHERE product_size = v_product_size);
                        WHILE v_new_sizes_to_add > 0 DO
                            INSERT INTO products VALUES (
                                (SELECT MAX(product_id) + 1 FROM products),
                                'Extra ' || v_product_size,
                                'General',
                                29.99,
                                19.99,
                                (SELECT MIN(supplier_id) FROM suppliers),
                                'Extra Contact',
                                'Extra Role',
                                '123 Extra St',
                                4,
                                'Red',
                                v_product_size,
                                0.5,
                                'Extra size',
                                4.0,
                                0,
                                CURRENT_TIMESTAMP,
                                CURRENT_TIMESTAMP,
                                true,
                                0.0,
                                0.19
                            );
                            SET v_new_sizes_to_add = v_new_sizes_to_add - 1;
                        END WHILE;
                    END;
                END IF;
            END FOR;
        END L2;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, CASE, WHILE, DML, deep-nesting, exception-handler, size-diversity-logic
-- EXPECTED: Product size balancing, inserts, size coverage enforcement, DML
-- EXECUTES: FOR, IF, CASE, WHILE, nested BEGIN...END, exception handler

-- Script 53
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_country STRING;
    DECLARE v_country_order_count INT;
    DECLARE v_country_supplier_count INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Country diversity error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_orders_per_country INT DEFAULT 1;
        DECLARE v_min_suppliers_per_country INT DEFAULT 1;
        FOR country AS SELECT DISTINCT customer_country FROM customers LIMIT 3 DO
            SET v_customer_country = country.customer_country;
            SET v_country_order_count = (SELECT COUNT(*) FROM orders WHERE customer_id IN (SELECT customer_id FROM customers WHERE customer_country = v_customer_country));
            IF v_country_order_count < v_min_orders_per_country THEN
                L1: BEGIN
                    INSERT INTO orders (
                        order_id,
                        customer_id,
                        order_date,
                        order_status,
                        order_total,
                        order_tax,
                        order_discount,
                        order_shipping
                    ) VALUES (
                        (SELECT MAX(order_id) + 1 FROM orders),
                        (SELECT MIN(customer_id) FROM customers WHERE customer_country = v_customer_country),
                        CURRENT_TIMESTAMP,
                        'Country Fill',
                        20.0,
                        2.0,
                        0.0,
                        2.0
                    );
                END L1;
            END IF;
            BEGIN
                DECLARE v_country_customer_count INT;
                SET v_country_customer_count = (SELECT COUNT(*) FROM customers WHERE customer_country = v_customer_country);
                IF v_country_customer_count < 2 THEN
                    L2: BEGIN
                        INSERT INTO customers VALUES (
                            (SELECT MAX(customer_id) + 1 FROM customers),
                            'Country ' || v_customer_country,
                            'country_' || LOWER(v_customer_country) || '@example.com',
                            '555-0000',
                            '123 Country St',
                            'Country City',
                            v_customer_country,
                            '00000',
                            v_customer_country,
                            'Retail',
                            CURRENT_TIMESTAMP,
                            0.0,
                            NULL,
                            NULL,
                            NULL,
                            true,
                            true,
                            0.0,
                            NULL,
                            NULL
                        );
                    END L2;
                END IF;
            END;
            SET v_country_supplier_count = (SELECT COUNT(*) FROM suppliers WHERE supplier_country = v_customer_country);
            IF v_country_supplier_count < v_min_suppliers_per_country THEN
                L3: BEGIN
                    INSERT INTO suppliers VALUES (
                        (SELECT MAX(supplier_id) + 1 FROM suppliers),
                        v_customer_country || ' Supplier',
                        'Supplier Contact',
                        'Supplier Role',
                        'supplier@' || LOWER(v_customer_country) || '.com',
                        '555-0001',
                        '123 Supplier St',
                        'Supplier City',
                        v_customer_country,
                        '00000',
                        v_customer_country,
                        v_customer_country,
                        'General',
                        'http://supplier.com',
                        CURRENT_TIMESTAMP,
                        NULL,
                        NULL,
                        4.0,
                        true
                    );
                END L3;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, country-diversity
-- EXPECTED: Order, customer, and supplier inserts, country diversity, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 54
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_weight DOUBLE;
    DECLARE v_product_category STRING;
    DECLARE v_category_weights INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Category weight error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_category_weight_count INT DEFAULT 3;
        FOR category AS SELECT DISTINCT product_category FROM products LIMIT 3 DO
            SET v_product_category = category.product_category;
            SET v_category_weights = (SELECT COUNT(*) FROM products WHERE product_category = v_product_category AND product_weight IS NOT NULL);
            IF v_category_weights < v_min_category_weight_count THEN
                L1: BEGIN
                BEGIN
                        DECLARE v_weight_to_set DOUBLE;
                        SET v_weight_to_set = 0.5 + (SELECT COUNT(*) FROM products WHERE product_category = v_product_category);
                        WHILE v_weight_to_set < 2.5 DO
                            INSERT INTO products VALUES (
                                (SELECT MAX(product_id) + 1 FROM products),
                                v_product_category || ' Weight',
                                v_product_category,
                                39.99,
                                29.99,
                                (SELECT MIN(supplier_id) FROM suppliers),
                                'Weight Contact',
                                'Weight Role',
                                '123 Weight St',
                                4,
                                CASE WHEN MOD((SELECT COUNT(*) FROM products WHERE product_category = v_product_category), 2) = 0 THEN 'Black' ELSE 'Gray' END,
                                'Standard',
                                v_weight_to_set,
                                'Weighted item',
                                3.0,
                                0,
                                CURRENT_TIMESTAMP,
                                CURRENT_TIMESTAMP,
                                true,
                                0.0,
                                0.19
                            );
                            SET v_weight_to_set = v_weight_to_set + 0.2;
                            SET v_category_weights = v_category_weights + 1;
                        END WHILE;
                    END;
                END L1;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, WHILE, DML, deep-nesting, exception-handler, weight-diversity
-- EXPECTED: Product weight inserts, weight diversity, DML
-- EXECUTES: FOR, IF, WHILE, nested BEGIN...END, exception handler

-- Script 55
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_opt_in BOOLEAN;
    DECLARE v_opt_in_count INT;
    DECLARE v_no_opt_in_count INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Opt-in error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        SET v_opt_in_count = (SELECT COUNT(*) FROM customers WHERE customer_opt_in_newsletter = true);
        SET v_no_opt_in_count = (SELECT COUNT(*) FROM customers WHERE customer_opt_in_newsletter = false);
        IF v_no_opt_in_count > v_opt_in_count THEN
            L1: BEGIN
                FOR customer AS SELECT customer_id FROM customers WHERE customer_opt_in_newsletter = false LIMIT 2 DO
                    IF v_no_opt_in_count > v_opt_in_count THEN
                    BEGIN
                        UPDATE customers
                        SET customer_opt_in_newsletter = true
                        WHERE customer_id = customer.customer_id;
                        SET v_opt_in_count = v_opt_in_count + 1;
                        SET v_no_opt_in_count = v_no_opt_in_count - 1;
                    END;
                    ELSE
                    BEGIN
                        DECLARE v_new_customer_id INT;
                        SET v_new_customer_id = (SELECT MAX(customer_id) + 1 FROM customers);
                        INSERT INTO customers VALUES (
                            v_new_customer_id,
                            'New Subscriber',
                            'subscriber@example.com',
                            '555-0002',
                            '123 Subscriber St',
                            'Newsletterville',
                            'CA',
                            '00000',
                            'USA',
                            'Retail',
                            CURRENT_TIMESTAMP,
                            0.0,
                            NULL,
                            NULL,
                            NULL,
                            true,
                            true,
                            0.0,
                            NULL,
                            NULL
                        );
                        SET v_opt_in_count = v_opt_in_count + 1;
                    END;
                    END IF;
                END FOR;
            END L1;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, opt-in-logic
-- EXPECTED: Opt-in updates, customer inserts, opt-in balance, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 56
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_city STRING;
    DECLARE v_city_customer_age INT;
    DECLARE v_average_customer_age DOUBLE;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Customer age diversity error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_cities_for_age_diversity INT DEFAULT 3;
        FOR city AS SELECT DISTINCT customer_city FROM customers LIMIT v_min_cities_for_age_diversity DO
            SET v_customer_city = city.customer_city;
            BEGIN
                SET v_average_customer_age = (SELECT AVG(DATEDIFF(YEAR, customer_birth_date, CURRENT_TIMESTAMP)) FROM customers WHERE customer_city = v_customer_city);
                IF v_average_customer_age IS NULL THEN
                    INSERT INTO customers VALUES (
                        (SELECT MAX(customer_id) + 1 FROM customers),
                        'Balanced City Customer',
                        'balanced@city.com',
                        '555-0001',
                        '123 Balance St',
                        v_customer_city,
                        'CA',
                        '00000',
                        'USA',
                        'Retail',
                        CURRENT_TIMESTAMP,
                        0.0,
                        NULL,
                        DATEADD(YEAR, -35, CURRENT_TIMESTAMP),
                        NULL,
                        true,
                        true,
                        0.0,
                        NULL,
                        NULL
                    );
                    SET v_average_customer_age = (SELECT AVG(DATEDIFF(YEAR, customer_birth_date, CURRENT_TIMESTAMP)) FROM customers WHERE customer_city = v_customer_city);
                END IF;
                IF v_average_customer_age < 25 THEN
                    INSERT INTO customers VALUES (
                        (SELECT MAX(customer_id) + 1 FROM customers),
                        'Older City Customer',
                        'older@city.com',
                        '555-0002',
                        '123 Senior St',
                        v_customer_city,
                        'CA',
                        '00000',
                        'USA',
                        'Retail',
                        CURRENT_TIMESTAMP,
                        0.0,
                        NULL,
                        DATEADD(YEAR, -50, CURRENT_TIMESTAMP),
                        NULL,
                        true,
                        true,
                        0.0,
                        NULL,
                        NULL
                    );
                    SET v_average_customer_age = (SELECT AVG(DATEDIFF(YEAR, customer_birth_date, CURRENT_TIMESTAMP)) FROM customers WHERE customer_city = v_customer_city);
                END IF;
                IF v_average_customer_age > 40 THEN
                    INSERT INTO customers VALUES (
                        (SELECT MAX(customer_id) + 1 FROM customers),
                        'Younger City Customer',
                        'younger@city.com',
                        '555-0003',
                        '123 Junior St',
                        v_customer_city,
                        'CA',
                        '00000',
                        'USA',
                        'Retail',
                        CURRENT_TIMESTAMP,
                        0.0,
                        NULL,
                        DATEADD(YEAR, -20, CURRENT_TIMESTAMP),
                        NULL,
                        true,
                        true,
                        0.0,
                        NULL,
                        NULL
                    );
                    SET v_average_customer_age = (SELECT AVG(DATEDIFF(YEAR, customer_birth_date, CURRENT_TIMESTAMP)) FROM customers WHERE customer_city = v_customer_city);
                END IF;
            END;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, age-diversity
-- EXPECTED: Customer age inserts, city age balancing, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 57
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_review DOUBLE;
    DECLARE v_product_rating DOUBLE;
    DECLARE v_reviews_per_category INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Product review/rating error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        DECLARE v_min_reviews_per_category INT DEFAULT 1;
        FOR category AS SELECT DISTINCT product_category FROM products LIMIT 3 DO
            SET v_reviews_per_category = (SELECT SUM(product_reviews) FROM products WHERE product_category = category.product_category);
            SET v_product_rating = (SELECT AVG(product_rating) FROM products WHERE product_category = category.product_category);
            L1: BEGIN
                IF v_reviews_per_category < v_min_reviews_per_category THEN
                    BEGIN
                        INSERT INTO products VALUES (
                            (SELECT MAX(product_id) + 1 FROM products),
                            'Review Filler',
                            category.product_category,
                            39.99,
                            29.99,
                            (SELECT MIN(supplier_id) FROM suppliers),
                            'Review Contact',
                            'Review Role',
                            '123 Reviews St',
                            6,
                            'Blue',
                            'Large',
                            1.2,
                            'Filled with reviews',
                            4.0,
                            25,
                            CURRENT_TIMESTAMP,
                            CURRENT_TIMESTAMP,
                            true,
                            0.0,
                            0.19
                        );
                        SET v_reviews_per_category = v_reviews_per_category + 25;
                    END;
                END IF;
                IF v_product_rating IS NULL OR v_product_rating < 3.0 THEN
                    BEGIN
                        UPDATE products
                        SET product_rating = 4.0
                        WHERE product_category = category.product_category AND product_reviews > 0;
                        SET v_product_rating = (SELECT AVG(product_rating) FROM products WHERE product_category = category.product_category);
                    END;
                END IF;
            END L1;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, review-diversity
-- EXPECTED: Product review and rating updates, review coverage, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 58
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_id INT;
    DECLARE v_product_description STRING;
    DECLARE v_description_length INT;
    DECLARE v_descriptive_product_count INT DEFAULT 0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Description diversity error'); END;

    L0: BEGIN
        DECLARE v_min_description_length INT DEFAULT 20;
        DECLARE v_max_description_length INT DEFAULT 100;
        DECLARE v_target_descriptive_products INT DEFAULT 5;
        FOR product AS SELECT product_id, product_description FROM products LIMIT 3 DO
            SET v_product_id = product.product_id;
            SET v_product_description = product.product_description;
            SET v_description_length = LENGTH(v_product_description);
            CASE
                WHEN v_description_length < v_min_description_length THEN
                    L1: BEGIN
                        DECLARE v_word_count INT DEFAULT v_min_description_length - v_description_length;
                        DECLARE v_new_description STRING;
                        REPEAT
                            SET v_new_description = CONCAT(v_product_description, ' detail');
                            SET v_word_count = v_word_count - 1;
                        UNTIL v_word_count <= 0
                        END REPEAT;
                        UPDATE products
                        SET product_description = v_new_description
                        WHERE product_id = v_product_id;
                        SET v_descriptive_product_count = v_descriptive_product_count + 1;
                    END L1;
                WHEN v_description_length > v_max_description_length THEN
                    BEGIN
                        UPDATE products
                        SET product_description = SUBSTRING(product_description, 1, v_max_description_length)
                        WHERE product_id = v_product_id;
                    END;
                ELSE
                    SET v_descriptive_product_count = v_descriptive_product_count + 1;
            END CASE;
        END FOR;
        IF v_descriptive_product_count < v_target_descriptive_products THEN
            L2: BEGIN
                DECLARE v_new_product_num INT DEFAULT v_target_descriptive_products - v_descriptive_product_count;
                REPEAT
                    INSERT INTO products VALUES (
                        (SELECT MAX(product_id) + 1 FROM products),
                        'Item with Long Description',
                        'General',
                        29.99,
                        19.99,
                        (SELECT MIN(supplier_id) FROM suppliers),
                        'Description Contact',
                        'Description Role',
                        '123 Desc St',
                        4,
                        'Blue',
                        'Medium',
                        0.8,
                        'This product has a long, detailed, and diverse description to maintain variability in the dataset.',
                        4.0,
                        0,
                        CURRENT_TIMESTAMP,
                        CURRENT_TIMESTAMP,
                        true,
                        0.0,
                        0.19
                    );
                    SET v_new_product_num = v_new_product_num - 1;
                UNTIL v_new_product_num <= 0
                END REPEAT;
            END L2;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, CASE, REPEAT, IF, DML, deep-nesting, exception-handler, description-diversity
-- EXPECTED: Product description diversity, updates and inserts, DML
-- EXECUTES: FOR, CASE, REPEAT, IF, nested BEGIN...END, exception handler

-- Script 59
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_item_status_nulls INT DEFAULT 0;
    DECLARE v_order_id INT;
    DECLARE v_order_item_id INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Order item status error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        SET v_order_item_status_nulls = (SELECT COUNT(*) FROM order_items WHERE item_status IS NULL);
        IF v_order_item_status_nulls > 0 THEN
            L1: BEGIN
                FOR order_item AS SELECT order_id, order_item_id FROM order_items WHERE item_status IS NULL LIMIT 2 DO
                    SET v_order_id = order_item.order_id;
                    SET v_order_item_id = order_item.order_item_id;
                    BEGIN
                        DECLARE v_order_status STRING;
                        SET v_order_status = (SELECT order_status FROM orders WHERE order_id = v_order_id);
                        CASE v_order_status
                            WHEN 'Completed' THEN
                                UPDATE order_items
                                SET item_status = 'Shipped'
                                WHERE order_item_id = v_order_item_id AND order_id = v_order_id;
                            WHEN 'Processing' THEN
                                UPDATE order_items
                                SET item_status = 'Preparing'
                                WHERE order_item_id = v_order_item_id AND order_id = v_order_id;
                            ELSE
                                UPDATE order_items
                                SET item_status = 'Received'
                                WHERE order_item_id = v_order_item_id AND order_id = v_order_id;
                        END CASE;
                    END;
                END FOR;
                SET v_order_item_status_nulls = (SELECT COUNT(*) FROM order_items WHERE item_status IS NULL);
                IF v_order_item_status_nulls > 1 THEN
                    REPEAT
                        UPDATE order_items
                        SET item_status = 'Cancelled'
                        WHERE order_item_id = (SELECT MIN(order_item_id) FROM order_items WHERE item_status IS NULL);
                        SET v_order_item_status_nulls = (SELECT COUNT(*) FROM order_items WHERE item_status IS NULL);
                    UNTIL v_order_item_status_nulls <= 1
                    END REPEAT;
                END IF;
            END L1;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, CASE, IF, REPEAT, DML, deep-nesting, exception-handler, status-diversity
-- EXPECTED: Item status filling, status logic, DML
-- EXECUTES: FOR, CASE, IF, REPEAT, nested BEGIN...END, exception handler

-- Script 60
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_is_prime BOOLEAN DEFAULT false;
    DECLARE v_prime_customer_count INT;
    DECLARE v_min_prime_customers INT DEFAULT 2;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Prime customer error'); END;

    L0: BEGIN
        SET v_prime_customer_count = (SELECT COUNT(*) FROM customers WHERE customer_is_prime_member = true);
        IF v_prime_customer_count < v_min_prime_customers THEN
            FOR cust AS SELECT customer_id FROM customers WHERE customer_is_prime_member IS NULL LIMIT v_min_prime_customers DO
                UPDATE customers
                SET customer_is_prime_member = true,
                    customer_prime_customer_since = CURRENT_TIMESTAMP
                WHERE customer_id = cust.customer_id;
                SET v_customer_is_prime = true;
            END FOR;
            IF v_customer_is_prime = false AND v_prime_customer_count < v_min_prime_customers THEN
                INSERT INTO customers VALUES (
                    (SELECT MAX(customer_id) + 1 FROM customers),
                    'Prime Customer',
                    'prime@example.com',
                    '555-0000',
                    '123 Prime St',
                    'Prime City',
                    'CA',
                    '00000',
                    'USA',
                    'Retail',
                    CURRENT_TIMESTAMP,
                    0.0,
                    NULL,
                    NULL,
                    NULL,
                    true,
                    true,
                    0.0,
                    CURRENT_TIMESTAMP,
                    NULL,
                    true
                );
                SET v_customer_is_prime = true;
            END IF;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, prime-customer-logic
-- EXPECTED: Prime customer status updates/inserts, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 61
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_employee_department STRING;
    DECLARE v_department_commission_total DOUBLE;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Department commission error'); END;

    -- Wide, deep, random logic, labels
    L0: BEGIN
        FOR dept AS SELECT DISTINCT employee_department FROM employees LIMIT 2 DO
            SET v_employee_department = dept.employee_department;
            SET v_department_commission_total = (SELECT SUM(employee_commission) FROM employees WHERE employee_department = v_employee_department);
            BEGIN
                DECLARE v_commission_threshold DOUBLE DEFAULT 2000;
                IF v_department_commission_total > v_commission_threshold THEN
                    L1: BEGIN
                        DECLARE v_commission_excess DOUBLE;
                        SET v_commission_excess = v_department_commission_total - v_commission_threshold;
                        WHILE v_commission_excess > 0 DO
                            SET v_commission_threshold = v_commission_threshold + 100;
                            UPDATE employees
                            SET employee_commission = employee_commission * 0.9
                            WHERE employee_department = v_employee_department;
                            SET v_department_commission_total = (SELECT SUM(employee_commission) FROM employees WHERE employee_department = v_employee_department);
                            SET v_commission_excess = v_department_commission_total - v_commission_threshold;
                        END WHILE;
                    END L1;
                END IF;
            END;
        END FOR;
        BEGIN
            DECLARE v_no_commission_employees INT;
            SET v_no_commission_employees = (SELECT COUNT(*) FROM employees WHERE employee_commission IS NULL OR employee_commission = 0);
            IF v_no_commission_employees > 0 THEN
                REPEAT
                    UPDATE employees
                    SET employee_commission = 100
                    WHERE employee_id = (SELECT MIN(employee_id) FROM employees WHERE employee_commission IS NULL OR employee_commission = 0);
                    SET v_no_commission_employees = v_no_commission_employees - 1;
                UNTIL v_no_commission_employees <= 1
                END REPEAT;
            END IF;
        END;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, WHILE, REPEAT, DML, deep-nesting, exception-handler, commission-diversity
-- EXPECTED: Commission balancing, updates, null handling, DML
-- EXECUTES: FOR, IF, WHILE, REPEAT, nested BEGIN...END, exception handler

-- Script 62
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_item_customizable_flag BOOLEAN DEFAULT false;
    DECLARE v_customizable_order_items_count INT;
    DECLARE v_min_customizable_items INT DEFAULT 2;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Customizable item error'); END;

    L0: BEGIN
        SET v_customizable_order_items_count = (SELECT COUNT(*) FROM order_items WHERE item_customization IS NOT NULL);
        IF v_customizable_order_items_count < v_min_customizable_items THEN
            FOR oi AS SELECT order_item_id FROM order_items WHERE item_customization IS NULL LIMIT v_min_customizable_items DO
                UPDATE order_items
                SET item_customization = CASE
                                            WHEN MOD(oi.order_item_id, 2) = 0 THEN 'Engraved'
                                            ELSE 'Monogrammed'
                                        END
                WHERE order_item_id = oi.order_item_id;
                SET v_order_item_customizable_flag = true;
            END FOR;
            IF v_order_item_customizable_flag = false AND v_customizable_order_items_count < v_min_customizable_items THEN
                INSERT INTO order_items (
                    order_item_id,
                    order_id,
                    product_id,
                    item_quantity,
                    item_price_per_unit,
                    item_discount,
                    item_tax,
                    item_total,
                    item_customization
                ) VALUES (
                    (SELECT MAX(order_item_id) + 1 FROM order_items),
                    (SELECT MIN(order_id) FROM orders),
                    (SELECT MIN(product_id) FROM products),
                    1,
                    10.0,
                    0.0,
                    1.0,
                    11.0,
                    'Engraved'
                );
                INSERT INTO order_items (
                    order_item_id,
                    order_id,
                    product_id,
                    item_quantity,
                    item_price_per_unit,
                    item_discount,
                    item_tax,
                    item_total,
                    item_customization
                ) VALUES (
                    (SELECT MAX(order_item_id) + 1 FROM order_items),
                    (SELECT MIN(order_id) FROM orders),
                    (SELECT MIN(product_id) FROM products),
                    1,
                    10.0,
                    0.0,
                    1.0,
                    11.0,
                    'Monogrammed'
                );
            END IF;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, customizable-item-logic
-- EXPECTED: Customized order item updates/inserts, item personalization, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 63
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_status STRING;
    DECLARE v_orders_with_items INT;
    DECLARE v_empty_orders INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Order-item linkage error'); END;

    L0: BEGIN
        SET v_orders_with_items = (SELECT COUNT(DISTINCT oi.order_id) FROM order_items oi);
        SET v_empty_orders = (SELECT COUNT(*) FROM orders o WHERE NOT EXISTS (SELECT 1 FROM order_items oi WHERE oi.order_id = o.order_id));
        IF v_empty_orders > 0 THEN
            L1: BEGIN
                FOR empty_order AS SELECT order_id FROM orders o WHERE NOT EXISTS (SELECT 1 FROM order_items oi WHERE oi.order_id = o.order_id) LIMIT 2 DO
                    SET v_order_status = (SELECT order_status FROM orders WHERE order_id = empty_order.order_id);
                    L2: BEGIN
                        CASE
                            WHEN v_order_status = 'Completed' THEN
                                INSERT INTO order_items (
                                    order_item_id,
                                    order_id,
                                    product_id,
                                    item_quantity,
                                    item_price_per_unit,
                                    item_discount,
                                    item_tax,
                                    item_total
                                ) VALUES (
                                    (SELECT MAX(order_item_id) + 1 FROM order_items),
                                    empty_order.order_id,
                                    (SELECT MIN(product_id) FROM products),
                                    1,
                                    10.0,
                                    0.0,
                                    1.0,
                                    11.0
                                );
                            WHEN v_order_status = 'Processing' THEN
                                INSERT INTO order_items (
                                    order_item_id,
                                    order_id,
                                    product_id,
                                    item_quantity,
                                    item_price_per_unit,
                                    item_discount,
                                    item_tax,
                                    item_total,
                                    item_status
                                ) VALUES (
                                    (SELECT MAX(order_item_id) + 1 FROM order_items),
                                    empty_order.order_id,
                                    (SELECT MIN(product_id) FROM products),
                                    1,
                                    10.0,
                                    0.0,
                                    1.0,
                                    11.0,
                                    'Preparing'
                                );
                            ELSE
                                BEGIN
                                    DECLARE v_product_id INT;
                                    SET v_product_id = (SELECT product_id FROM products WHERE product_category = 'Specialty' LIMIT 1);
                                    IF v_product_id IS NULL THEN
                                        SET v_product_id = (SELECT MIN(product_id) FROM products);
                                    END IF;
                                    INSERT INTO order_items (
                                        order_item_id,
                                        order_id,
                                        product_id,
                                        item_quantity,
                                        item_price_per_unit,
                                        item_discount,
                                        item_tax,
                                        item_total,
                                        item_status
                                    ) VALUES (
                                        (SELECT MAX(order_item_id) + 1 FROM order_items),
                                        empty_order.order_id,
                                        v_product_id,
                                        1,
                                        20.0,
                                        0.0,
                                        2.0,
                                        22.0,
                                        CASE
                                            WHEN RANDOM() > 0.5 THEN 'Shipped'
                                            ELSE 'Processing'
                                        END
                                    );
                                END;
                        END CASE;
                    END L2;
                    SET v_empty_orders = v_empty_orders - 1;
                END FOR;
            END L1;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, CASE, DML, deep-nesting, exception-handler, order-item-linkage
-- EXPECTED: Order-item binding, status-based item creation, DML
-- EXECUTES: FOR, IF, CASE, nested BEGIN...END, exception handler

-- Script 64
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_supplier_site_count INT;
    DECLARE v_site_char_length INT;
    DECLARE v_min_site_length INT DEFAULT 8;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Supplier site error'); END;

    L0: BEGIN
        SET v_supplier_site_count = (SELECT COUNT(*) FROM suppliers WHERE supplier_website IS NOT NULL);
        IF v_supplier_site_count < 5 THEN
            L1: BEGIN
                FOR s1 AS SELECT supplier_id FROM suppliers WHERE supplier_website IS NULL OR supplier_website = '' LIMIT 3 DO
                    SET v_site_char_length = (SELECT LENGTH(supplier_website) FROM suppliers WHERE supplier_id = s1.supplier_id);
                    IF v_site_char_length < v_min_site_length THEN
                        L2: BEGIN
                            IF EXISTS (SELECT 1 FROM suppliers WHERE supplier_id = s1.supplier_id) THEN
                                UPDATE suppliers
                                SET supplier_website = CONCAT('https://', REPLACE(LOWER(supplier_name), ' ', ''), '.com')
                                WHERE supplier_id = s1.supplier_id;
                            END IF;
                        END L2;
                    END IF;
                    SET v_supplier_site_count = (SELECT COUNT(*) FROM suppliers WHERE supplier_website IS NOT NULL);
                    IF v_supplier_site_count >= 5 THEN
                        LEAVE L1;
                    END IF;
                END FOR;
            END L1;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, LEAVE, DML, deep-nesting, exception-handler, website-logic
-- EXPECTED: Supplier website diversity, format updates, DML
-- EXECUTES: FOR, IF, LEAVE, nested BEGIN...END, exception handler

-- Script 65
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_units_sold DOUBLE;
    DECLARE v_category_products_sold INT;
    DECLARE v_sale_regrowth BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Inventory movement error'); END;

    L0: BEGIN
        FOR cat AS SELECT DISTINCT product_category FROM products LIMIT 3 DO
            SET v_product_units_sold = (SELECT SUM(oi.item_quantity) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE p.product_category = cat.product_category);
            SET v_category_products_sold = (SELECT COUNT(DISTINCT p.product_id) FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE p.product_category = cat.product_category);
            IF v_product_units_sold < 10 THEN
                L1: BEGIN
                    WHILE v_product_units_sold < 20 AND v_category_products_sold < 2 DO
                        INSERT INTO products VALUES (
                            (SELECT MAX(product_id) + 1 FROM products),
                            'Fast-Moving ' || cat.product_category,
                            cat.product_category,
                            35.99,
                            25.99,
                            (SELECT MIN(supplier_id) FROM suppliers WHERE supplier_main_products LIKE '%' || cat.product_category || '%'),
                            'Movement Contact',
                            'Movement Role',
                            '123 Moving St',
                            8,
                            'Any',
                            'Any',
                            0.7,
                            'Fast mover',
                            4.5,
                            0,
                            CURRENT_TIMESTAMP,
                            CURRENT_TIMESTAMP,
                            true,
                            0.0,
                            0.19
                        );
                        SET v_category_products_sold = v_category_products_sold + 1;
                        INSERT INTO order_items (
                            order_item_id,
                            order_id,
                            product_id,
                            item_quantity,
                            item_price_per_unit,
                            item_discount,
                            item_tax,
                            item_total
                        ) VALUES (
                            (SELECT MAX(order_item_id) + 1 FROM order_items),
                            (SELECT MIN(order_id) FROM orders),
                            (SELECT MAX(product_id) FROM products WHERE product_category = cat.product_category),
                            2,
                            35.99,
                            0.0,
                            2.0,
                            37.99
                        );
                        SET v_product_units_sold = v_product_units_sold + 2;
                    END WHILE;
                    SET v_sale_regrowth = true;
                END L1;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, WHILE, DML, deep-nesting, exception-handler, sales-velocity
-- EXPECTED: Product and sales velocity inserts, category balancing, DML
-- EXECUTES: FOR, WHILE, nested BEGIN...END, exception handler

-- Script 66
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_supplier_product_match INT;
    DECLARE v_is_supplier_core BOOLEAN;
    DECLARE v_product_supplier_missing BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Supplier-product match error'); END;

    L0: BEGIN
        FOR sup AS SELECT supplier_id, supplier_main_products FROM suppliers LIMIT 2 DO
            SET v_supplier_product_match = (SELECT COUNT(*) FROM products p WHERE p.supplier_id = sup.supplier_id AND p.product_category = sup.supplier_main_products);
            IF v_supplier_product_match = 0 THEN
                L1: BEGIN
                    SET v_is_supplier_core = (sup.supplier_main_products IS NOT NULL AND sup.supplier_main_products != '');
                    IF v_is_supplier_core THEN
                        INSERT INTO products VALUES (
                            (SELECT MAX(product_id) + 1 FROM products),
                            sup.supplier_main_products || ' Special',
                            sup.supplier_main_products,
                            44.99,
                            34.99,
                            sup.supplier_id,
                            sup.supplier_name,
                            'Product Specialist',
                            '123 Main St',
                            10,
                            'Any',
                            'Large',
                            1.3,
                            'Supplier-matched',
                            4.2,
                            0,
                            CURRENT_TIMESTAMP,
                            CURRENT_TIMESTAMP,
                            true,
                            0.0,
                            0.19
                        );
                    END IF;
                END L1;
            END IF;
        END FOR;
    END L0;

    L2: BEGIN
        IF EXISTS (
            SELECT 1
            FROM products
            WHERE product_category NOT LIKE '%' || (
                SELECT supplier_main_products
                FROM suppliers
                WHERE supplier_id = products.supplier_id
            ) || '%'
            AND supplier_id IS NOT NULL
        ) THEN
            FOR prod AS SELECT product_id, product_category, supplier_id FROM products WHERE product_category NOT LIKE '%' || (
                SELECT supplier_main_products
                FROM suppliers
                WHERE supplier_id = products.supplier_id
            ) || '%' LIMIT 2 DO
                IF EXISTS (
                    SELECT 1
                    FROM suppliers
                    WHERE supplier_main_products LIKE '%' || prod.product_category || '%'
                ) THEN
                    UPDATE products
                    SET supplier_id = (
                        SELECT MIN(supplier_id)
                        FROM suppliers
                        WHERE supplier_main_products LIKE '%' || prod.product_category || '%'
                    )
                    WHERE product_id = prod.product_id;
                END IF;
            END FOR;
        END IF;
    END L2;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, FOREACH, DML, deep-nesting, exception-handler, supplier-product-matching
-- EXPECTED: Supplier-product category matching, updates and inserts, DML
-- EXECUTES: FOR, IF, FOREACH, nested BEGIN...END, exception handler

-- Script 67
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_employee_tenure_years INT;
    DECLARE v_employee_salary DOUBLE;
    DECLARE v_salary_differential DOUBLE DEFAULT 0.1;
    DECLARE v_tenure_threshold INT DEFAULT 2;
    DECLARE v_tenure_based_adjustments INT DEFAULT 0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Tenure salary error'); END;

    L0: BEGIN
        FOR emp AS SELECT employee_id, employee_salary, employee_hire_date FROM employees LIMIT 3 DO
            IF emp.employee_hire_date IS NOT NULL THEN
                SET v_employee_tenure_years = DATEDIFF(YEAR, emp.employee_hire_date, CURRENT_TIMESTAMP);
                SET v_employee_salary = emp.employee_salary;
                -- Use a simple IF to avoid repetitive control structures
                IF v_employee_tenure_years > v_tenure_threshold THEN
                    UPDATE employees
                    SET employee_salary = ROUND(employee_salary * (1.0 + v_salary_differential), 2)
                    WHERE employee_id = emp.employee_id;
                    SET v_tenure_based_adjustments = v_tenure_based_adjustments + 1;
                END IF;
                -- Introduce some randomness and logic, but not with REPEAT/WHILE
                IF v_employee_tenure_years = v_tenure_threshold THEN
                    UPDATE employees
                    SET employee_salary = ROUND(employee_salary * 0.95, 2),
                        employee_description = CONCAT('Provisional (', v_employee_tenure_years, ' years)')
                    WHERE employee_id = emp.employee_id;
                END IF;
            END IF;
        END FOR;
        -- Insert a new employee for diversity, but not in a loop
        IF v_tenure_based_adjustments < 1 THEN
            INSERT INTO employees VALUES (
                (SELECT MAX(employee_id) + 1 FROM employees),
                'New Hire',
                'hire@company.com',
                '555-0000',
                'Associate',
                'Operations',
                CURRENT_TIMESTAMP,
                NULL,
                40000.0,
                0.0,
                NULL,
                '123 Hire St',
                NULL,
                'New City',
                'CA',
                '00000',
                'USA',
                CURRENT_TIMESTAMP,
                'Other',
                'Single',
                0
            );
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, salary-tenure-logic
-- EXPECTED: Tenure-based salary adjustments, diversity hire, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 68
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_shipping_region STRING;
    DECLARE v_region_order_count INT;
    DECLARE v_max_isolation_level INT DEFAULT 2;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Region isolation error'); END;

    L0: BEGIN
        FOR region AS SELECT DISTINCT order_shipping_region FROM orders LIMIT 3 DO
            SET v_order_shipping_region = region.order_shipping_region;
            SET v_region_order_count = (SELECT COUNT(*) FROM orders WHERE order_shipping_region = v_order_shipping_region);
            IF v_region_order_count < v_max_isolation_level THEN
                L1: BEGIN
                    SET v_region_order_count = (SELECT COUNT(*) FROM customers WHERE customer_country = v_order_shipping_region);
                    IF v_region_order_count = 0 THEN
                        INSERT INTO customers VALUES (
                            (SELECT MAX(customer_id) + 1 FROM customers),
                            'Region ' || v_order_shipping_region,
                            'region@regional.com',
                            '555-0000',
                            '123 Regional St',
                            v_order_shipping_region,
                            v_order_shipping_region,
                            '00000',
                            v_order_shipping_region,
                            'Wholesale',
                            CURRENT_TIMESTAMP,
                            0.0,
                            NULL,
                            NULL,
                            NULL,
                            true,
                            true,
                            0.0,
                            NULL,
                            NULL
                        );
                    END IF;
                    INSERT INTO orders (
                        order_id,
                        customer_id,
                        order_date,
                        order_status,
                        order_total,
                        order_tax,
                        order_discount,
                        order_shipping,
                        order_shipping_region
                    ) VALUES (
                        (SELECT MAX(order_id) + 1 FROM orders),
                        (SELECT MIN(customer_id) FROM customers WHERE customer_country = v_order_shipping_region),
                        CURRENT_TIMESTAMP,
                        'Regional Isolation',
                        30.0,
                        3.0,
                        0.0,
                        5.0,
                        v_order_shipping_region
                    );
                END L1;
            END IF;
        END FOR;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, region-isolation
-- EXPECTED: Regional order and customer inserts, geographical isolation, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 69
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_tax_percent DOUBLE;
    DECLARE v_country_tax_rates INT;
    DECLARE v_tax_exception BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Tax rate diversity error'); END;

    L0: BEGIN
        SET v_country_tax_rates = (
            SELECT COUNT(DISTINCT CONCAT(order_tax, order_shipping_country))
            FROM orders
            WHERE order_tax != 0
        );
        IF v_country_tax_rates < 4 AND (SELECT COUNT(DISTINCT order_shipping_country) FROM orders) > 2 THEN
            L1: BEGIN
                FOR country_tax_pair AS SELECT DISTINCT order_shipping_country FROM orders WHERE order_tax = 0 LIMIT 4 DO
                    SET v_order_tax_percent = CASE
                        WHEN country_tax_pair.order_shipping_country = 'USA' THEN 0.08
                        WHEN country_tax_pair.order_shipping_country = 'UK' THEN 0.20
                        WHEN country_tax_pair.order_shipping_country = 'CA' THEN 0.05
                        WHEN country_tax_pair.order_shipping_country = 'AU' THEN 0.10
                        ELSE 0.19
                    END;
                    INSERT INTO orders (
                        order_id,
                        customer_id,
                        order_date,
                        order_status,
                        order_total,
                        order_tax,
                        order_discount,
                        order_shipping,
                        order_shipping_country
                    ) VALUES (
                        (SELECT MAX(order_id) + 1 FROM orders),
                        (SELECT MIN(customer_id) FROM customers),
                        CURRENT_TIMESTAMP,
                        'Tax Diversity',
                        50.0,
                        ROUND(50.0 * v_order_tax_percent, 2),
                        0.0,
                        2.0,
                        country_tax_pair.order_shipping_country
                    );
                END FOR;
                SET v_tax_exception = true;
            END L1;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOREACH, IF, DML, deep-nesting, exception-handler, tax-diversity
-- EXPECTED: Country tax inserts, tax rate balancing, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 70
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_id INT;
    DECLARE v_product_reviews INT;
    DECLARE v_products_without_reviews INT;
    DECLARE v_review_threshold INT DEFAULT 5;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Review coverage error'); END;

    L0: BEGIN
        SET v_products_without_reviews = (SELECT COUNT(*) FROM products WHERE product_reviews = 0 OR product_reviews IS NULL);
        IF v_products_without_reviews > 0 THEN
            FOR product_no_reviews AS SELECT product_id FROM products WHERE product_reviews = 0 OR product_reviews IS NULL LIMIT 3 DO
                SET v_product_id = product_no_reviews.product_id;
                SET v_product_reviews = (SELECT product_reviews FROM products WHERE product_id = v_product_id);
                IF v_product_reviews < v_review_threshold THEN
                    SET v_product_reviews = v_product_reviews + 5;
                    UPDATE products
                    SET product_reviews = v_product_reviews,
                        product_description = CONCAT(product_description, ' | REVIEWED')
                    WHERE product_id = v_product_id;
                END IF;
            END FOR;
        ELSE
            -- To ensure future variety, inject a product with zero reviews
            INSERT INTO products VALUES (
                (SELECT MAX(product_id) + 1 FROM products),
                'Never Reviewed',
                'Misc',
                9.99,
                4.99,
                (SELECT MIN(supplier_id) FROM suppliers),
                'Noone',
                'NoContact',
                '123 No St',
                1,
                'Clear',
                'Small',
                0.1,
                'Waiting for first review',
                0.0,
                0,
                CURRENT_TIMESTAMP,
                CURRENT_TIMESTAMP,
                true,
                0.0,
                0.19
            );
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, review-coverage-logic
-- EXPECTED: Review counter updates, unreviewed product insertion, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 71
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_active_missing INT;
    DECLARE v_temp_active_set BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Active status error'); END;

    L0: BEGIN
        SET v_customer_active_missing = (SELECT COUNT(*) FROM customers WHERE customer_active_status IS NULL);
        IF v_customer_active_missing > 0 THEN
            FOR cust_missing_active AS SELECT customer_id FROM customers WHERE customer_active_status IS NULL LIMIT 3 DO
                UPDATE customers
                SET customer_active_status = (DATEDIFF(DAY, customer_membership_date, CURRENT_TIMESTAMP) < 365)
                WHERE customer_id = cust_missing_active.customer_id;
                SET v_temp_active_set = true;
            END FOR;
        END IF;
        IF v_temp_active_set = false THEN
            -- Force at least one unset active status for future handling
            INSERT INTO customers VALUES (
                (SELECT MAX(customer_id) + 1 FROM customers),
                'Unset Active',
                'inactive@example.com',
                '555-0000',
                '123 Unset St',
                'Unknown',
                'CA',
                '00000',
                'USA',
                'Retail',
                CURRENT_TIMESTAMP,
                0.0,
                NULL,
                NULL,
                NULL,
                NULL,
                true,
                0.0,
                NULL,
                NULL
            );
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, active-status-logic
-- EXPECTED: Customer active status setting, forced unset case, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 72
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_type_count INT;
    DECLARE v_low_diversity_type STRING;
    DECLARE v_orders_by_type INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Type order diversity error'); END;

    L0: BEGIN
        SET v_customer_type_count = (SELECT COUNT(DISTINCT customer_type) FROM customers);
        IF v_customer_type_count < 5 THEN
            FOR type_boost AS SELECT DISTINCT customer_type FROM customers LIMIT 3 DO
                SET v_orders_by_type = (SELECT COUNT(*) FROM orders o JOIN customers c ON o.customer_id = c.customer_id WHERE c.customer_type = type_boost.customer_type);
                IF v_orders_by_type < 2 THEN
                    INSERT INTO orders (
                        order_id,
                        customer_id,
                        order_date,
                        order_status,
                        order_total,
                        order_tax,
                        order_discount,
                        order_shipping
                    ) VALUES (
                        (SELECT MAX(order_id) + 1 FROM orders),
                        (SELECT MIN(customer_id) FROM customers WHERE customer_type = type_boost.customer_type),
                        CURRENT_TIMESTAMP,
                        'Type Filler',
                        15.0,
                        1.5,
                        0.0,
                        2.0
                    );
                END IF;
            END FOR;
        ELSE
            SET v_low_diversity_type = (SELECT customer_type FROM customers GROUP BY customer_type ORDER BY COUNT(*) LIMIT 1);
            IF (SELECT COUNT(*) FROM orders o JOIN customers c ON o.customer_id = c.customer_id WHERE c.customer_type = v_low_diversity_type) < 1 THEN
                INSERT INTO orders (
                    order_id,
                    customer_id,
                    order_date,
                    order_status,
                    order_total,
                    order_tax,
                    order_discount,
                    order_shipping
                ) VALUES (
                    (SELECT MAX(order_id) + 1 FROM orders),
                    (SELECT MIN(customer_id) FROM customers WHERE customer_type = v_low_diversity_type),
                    CURRENT_TIMESTAMP,
                    'Type Revival',
                    10.0,
                    1.0,
                    0.0,
                    1.0
                );
            END IF;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, type-diversity-logic
-- EXPECTED: Order inserts for customer type diversity, type balancing, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 73
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_item_offer_count INT;
    DECLARE v_offer_code_required BOOLEAN DEFAULT false;
    DECLARE v_order_item_missing_offer INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Offer code error'); END;

    L0: BEGIN
        SET v_order_item_offer_count = (SELECT COUNT(*) FROM order_items WHERE item_offer_code IS NOT NULL);
        IF v_order_item_offer_count < 2 THEN
            FOR order_item_no_offer AS SELECT order_item_id FROM order_items WHERE item_offer_code IS NULL LIMIT 2 DO
                UPDATE order_items
                SET item_offer_code = CASE
                                        WHEN RANDOM() > 0.7 THEN 'SPRING24'
                                        WHEN RANDOM() > 0.3 THEN 'FALLWINTER'
                                        ELSE 'SUMMER2025'
                                     END
                WHERE order_item_id = order_item_no_offer.order_item_id;
                SET v_order_item_offer_count = v_order_item_offer_count + 1;
            END FOR;
        END IF;
        SET v_order_item_missing_offer = (SELECT COUNT(*) FROM order_items WHERE item_offer_code IS NULL);
        IF v_order_item_missing_offer > 5 THEN
            UPDATE order_items
            SET item_offer_code = 'NOCODE'
            WHERE order_item_id = (SELECT MIN(order_item_id) FROM order_items WHERE item_offer_code IS NULL);
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, offer-diversity
-- EXPECTED: Offer code insertion, offer logic, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 74
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_employee_dept_stagnant INT;
    DECLARE v_dept_stagnant_name STRING;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Dept stagnation error'); END;

    L0: BEGIN
        SET v_employee_dept_stagnant = (
            SELECT COUNT(*)
            FROM employees
            WHERE employee_hire_date < DATEADD(YEAR, -5, CURRENT_TIMESTAMP)
              AND employee_department = (
                  SELECT employee_department
                  FROM employees
                  GROUP BY employee_department
                  ORDER BY AVG(DATEDIFF(YEAR, employee_hire_date, CURRENT_TIMESTAMP)) DESC
                  LIMIT 1
              )
        );
        IF v_employee_dept_stagnant > 1 THEN
            SET v_dept_stagnant_name = (
                SELECT employee_department
                FROM employees
                GROUP BY employee_department
                ORDER BY AVG(DATEDIFF(YEAR, employee_hire_date, CURRENT_TIMESTAMP)) DESC
                LIMIT 1
            );
            INSERT INTO employees VALUES (
                (SELECT MAX(employee_id) + 1 FROM employees),
                'Diverse Dept Member',
                'rotate@company.com',
                '555-0000',
                'Associate',
                v_dept_stagnant_name,
                CURRENT_TIMESTAMP,
                NULL,
                45000.0,
                0.0,
                NULL,
                '123 Diversion St',
                NULL,
                'New City',
                'CA',
                '00000',
                'USA',
                CURRENT_TIMESTAMP,
                'Other',
                'Single',
                0
            );
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: IF, DML, deep-nesting, exception-handler, department-logic
-- EXPECTED: Employee department balancing insert, department stagnation logic, DML
-- EXECUTES: IF, nested BEGIN...END, exception handler

-- Script 75
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_supplier_product_min INT DEFAULT 2;
    DECLARE v_products_per_supplier INT;
    DECLARE v_suppliers_with_stock INT DEFAULT 0;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Supplier-product linkage error'); END;

    L0: BEGIN
        FOR sup AS SELECT supplier_id FROM suppliers LIMIT 3 DO
            SET v_products_per_supplier = (SELECT COUNT(*) FROM products WHERE supplier_id = sup.supplier_id);
            IF v_products_per_supplier < v_supplier_product_min THEN
                BEGIN
                    DECLARE v_new_product_id INT;
                    SET v_new_product_id = (SELECT MAX(product_id) + 1 FROM products);
                    INSERT INTO products VALUES (
                        v_new_product_id,
                        'Supplier Stock',
                        'Generic',
                        24.99,
                        19.99,
                        sup.supplier_id,
                        'Stock Contact',
                        'Stock Role',
                        '123 Stock St',
                        5,
                        'White',
                        'Standard',
                        0.6,
                        'Supplier item',
                        3.0,
                        0,
                        CURRENT_TIMESTAMP,
                        CURRENT_TIMESTAMP,
                        true,
                        0.0,
                        0.19
                    );
                    SET v_suppliers_with_stock = v_suppliers_with_stock + 1;
                END;
            END IF;
        END FOR;
    END L0;
    BEGIN
        IF v_suppliers_with_stock < 2 AND (SELECT COUNT(*) FROM products WHERE supplier_id IS NULL) > 0 THEN
            UPDATE products
            SET supplier_id = (SELECT MIN(supplier_id) FROM suppliers)
            WHERE product_id = (SELECT MIN(product_id) FROM products WHERE supplier_id IS NULL);
        END IF;
    END;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, supplier-stock-logic
-- EXPECTED: Supplier-product linkage, product inserts, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 76
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_discount_high INT;
    DECLARE v_product_discount_low INT;
    DECLARE v_discount_skew_ratio DOUBLE;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Discount balancing error'); END;

    L0: BEGIN
        SET v_product_discount_high = (SELECT COUNT(*) FROM products WHERE product_discount > 0.2);
        SET v_product_discount_low = (SELECT COUNT(*) FROM products WHERE product_discount > 0 AND product_discount <= 0.2);
        SET v_discount_skew_ratio = v_product_discount_high / GREATEST(v_product_discount_low, 1.0);
        IF v_discount_skew_ratio > 2.0 THEN
            BEGIN
                DECLARE v_over_discounted_product_id INT;
                SET v_over_discounted_product_id = (SELECT MIN(product_id) FROM products WHERE product_discount > 0.2 LIMIT 1);
                IF v_over_discounted_product_id IS NOT NULL THEN
                    REPEAT
                        UPDATE products
                        SET product_discount = ROUND(product_discount * 0.9, 2)
                        WHERE product_id = v_over_discounted_product_id;
                        SET v_product_discount_high = (SELECT COUNT(*) FROM products WHERE product_discount > 0.2);
                        SET v_discount_skew_ratio = v_product_discount_high / GREATEST(v_product_discount_low, 1.0);
                    UNTIL v_product_discount_high <= 2 OR v_discount_skew_ratio <= 2.0 OR v_over_discounted_product_id IS NULL
                    END REPEAT;
                END IF;
            END;
        ELSEIF v_discount_skew_ratio < 0.5 AND v_product_discount_high < 2 THEN
            REPEAT
                INSERT INTO products VALUES (
                    (SELECT MAX(product_id) + 1 FROM products),
                    'Discount Blowout',
                    'Sale',
                    49.99,
                    34.99,
                    (SELECT MIN(supplier_id) FROM suppliers),
                    'Sale Contact',
                    'Sale Role',
                    '123 Sale St',
                    8,
                    'Red',
                    'Large',
                    1.1,
                    'Big sale',
                    4.0,
                    0,
                    CURRENT_TIMESTAMP,
                    CURRENT_TIMESTAMP,
                    true,
                    0.3,
                    0.19
                );
                SET v_product_discount_high = (SELECT COUNT(*) FROM products WHERE product_discount > 0.2);
            UNTIL v_product_discount_high >= 2 OR v_discount_skew_ratio >= 0.5
            END REPEAT;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: IF, REPEAT, DML, deep-nesting, exception-handler, discount-balancing
-- EXPECTED: Discount rate balancing, high-discount product control, DML
-- EXECUTES: IF, REPEAT, nested BEGIN...END, exception handler

-- Script 77
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_last_product_update_days INT;
    DECLARE v_products_needing_update INT;
    DECLARE v_freshness_target INT DEFAULT 7;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Product timestamp error'); END;

    L0: BEGIN
        SET v_last_product_update_days = (SELECT MAX(DATEDIFF(DAY, product_last_update, CURRENT_TIMESTAMP)) FROM products);
        SET v_products_needing_update = (SELECT COUNT(*) FROM products WHERE product_last_update IS NULL OR DATEDIFF(DAY, product_last_update, CURRENT_TIMESTAMP) > v_freshness_target);
        IF v_last_product_update_days > v_freshness_target AND v_products_needing_update > 0 THEN
            FOR product_stale AS SELECT product_id FROM products WHERE product_last_update IS NULL OR DATEDIFF(DAY, product_last_update, CURRENT_TIMESTAMP) > v_freshness_target LIMIT 3 DO
                UPDATE products
                SET product_last_update = CURRENT_TIMESTAMP,
                    product_description = CONCAT(product_description, ' | Fresh')
                WHERE product_id = product_stale.product_id;
            END FOR;
        END IF;
    END L0;

    BEGIN
        IF (SELECT COUNT(*) FROM products WHERE product_last_update > DATEADD(DAY, -3, CURRENT_TIMESTAMP)) < 2 THEN
            REPEAT
                INSERT INTO products VALUES (
                    (SELECT MAX(product_id) + 1 FROM products),
                    'Just In',
                    'New Arrivals',
                    59.99,
                    49.99,
                    (SELECT MIN(supplier_id) FROM suppliers),
                    'New Contact',
                    'New Role',
                    '123 Fresh St',
                    12,
                    'Green',
                    'Medium',
                    0.8,
                    'Fresh item',
                    4.5,
                    0,
                    CURRENT_TIMESTAMP,
                    CURRENT_TIMESTAMP,
                    true,
                    0.0,
                    0.19
                );
            UNTIL (SELECT COUNT(*) FROM products WHERE product_last_update > DATEADD(DAY, -3, CURRENT_TIMESTAMP)) >= 2
            END REPEAT;
        END IF;
    END;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, REPEAT, DML, deep-nesting, exception-handler, product-freshness
-- EXPECTED: Product freshness updates, fresh item inserts, DML
-- EXECUTES: FOR, REPEAT, nested BEGIN...END, exception handler

-- Script 78
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_total_min DOUBLE DEFAULT 10.0;
    DECLARE v_order_count_low_value INT;
    DECLARE v_order_min_value_id INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Order value error'); END;

    L0: BEGIN
        SET v_order_count_low_value = (SELECT COUNT(*) FROM orders WHERE order_total < v_order_total_min);
        IF v_order_count_low_value > 0 THEN
            L1: BEGIN
                SET v_order_min_value_id = (SELECT MIN(order_id) FROM orders WHERE order_total < v_order_total_min);
                REPEAT
                    UPDATE orders
                    SET order_total = v_order_total_min,
                        order_discount = 0.0
                    WHERE order_id = v_order_min_value_id;
                    SET v_order_min_value_id = (SELECT MIN(order_id) FROM orders WHERE order_total < v_order_total_min);
                UNTIL v_order_min_value_id IS NULL OR v_order_count_low_value < 2
                END REPEAT;
            END L1;
        ELSE
            L2: BEGIN
                INSERT INTO orders (
                    order_id,
                    customer_id,
                    order_date,
                    order_status,
                    order_total,
                    order_tax,
                    order_discount,
                    order_shipping
                ) VALUES (
                    (SELECT MAX(order_id) + 1 FROM orders),
                    (SELECT MIN(customer_id) FROM customers),
                    CURRENT_TIMESTAMP,
                    'Low Value',
                    v_order_total_min / 2.0,
                    0.0,
                    0.0,
                    1.0
                );
            END L2;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: IF, REPEAT, DML, deep-nesting, exception-handler, order-value-logic
-- EXPECTED: Order minimum value enforcement, low-value order creation, DML
-- EXECUTES: IF, REPEAT, nested BEGIN...END, exception handler

-- Script 79
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_employee_transfers_required INT DEFAULT 2;
    DECLARE v_longest_department_employees INT;
    DECLARE v_shortest_department_employees INT;
    DECLARE v_transfer_employee_id INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Department balance error'); END;

    L0: BEGIN
        SET v_longest_department_employees = (
            SELECT COUNT(*)
            FROM employees
            WHERE employee_department = (
                SELECT employee_department
                FROM employees
                GROUP BY employee_department
                ORDER BY AVG(DATEDIFF(YEAR, employee_hire_date, CURRENT_TIMESTAMP)) DESC
                LIMIT 1
            )
        );
        SET v_shortest_department_employees = (
            SELECT COUNT(*)
            FROM employees
            WHERE employee_department = (
                SELECT employee_department
                FROM employees
                GROUP BY employee_department
                ORDER BY AVG(DATEDIFF(YEAR, employee_hire_date, CURRENT_TIMESTAMP)) ASC
                LIMIT 1
            )
        );
        IF v_longest_department_employees > v_shortest_department_employees + 2 THEN
            L1: BEGIN
                SET v_transfer_employee_id = (
                    SELECT MIN(employee_id)
                    FROM employees
                    WHERE employee_department = (
                        SELECT employee_department
                        FROM employees
                        GROUP BY employee_department
                        ORDER BY AVG(DATEDIFF(YEAR, employee_hire_date, CURRENT_TIMESTAMP)) DESC
                        LIMIT 1
                    )
                );
                IF v_transfer_employee_id IS NOT NULL THEN
                    UPDATE employees
                    SET employee_department = (
                        SELECT employee_department
                        FROM employees
                        GROUP BY employee_department
                        ORDER BY AVG(DATEDIFF(YEAR, employee_hire_date, CURRENT_TIMESTAMP)) ASC
                        LIMIT 1
                    )
                    WHERE employee_id = v_transfer_employee_id;
                END IF;
            END L1;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: IF, DML, deep-nesting, exception-handler, department-balance-logic
-- EXPECTED: Department balancing via transfer, staffing logic, DML
-- EXECUTES: IF, nested BEGIN...END, exception handler

-- Script 80
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_supplier_region_imbalance INT;
    DECLARE v_most_imbalanced_region STRING;
    DECLARE v_region_sku_count INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Region sku error'); END;

    L0: BEGIN
        SET v_supplier_region_imbalance = (
            SELECT MAX(cnt) - MIN(cnt)
            FROM (
                SELECT COUNT(*) AS cnt
                FROM products p
                JOIN suppliers s ON p.supplier_id = s.supplier_id
                GROUP BY s.supplier_region
            ) t
        );
        IF v_supplier_region_imbalance > 3 THEN
            SET v_most_imbalanced_region = (
                SELECT supplier_region
                FROM suppliers s
                JOIN products p ON s.supplier_id = p.supplier_id
                GROUP BY s.supplier_region
                HAVING COUNT(*) = (
                    SELECT MIN(cnt)
                    FROM (
                        SELECT COUNT(*) AS cnt
                        FROM products p
                        JOIN suppliers s ON p.supplier_id = s.supplier_id
                        GROUP BY s.supplier_region
                    ) t
                )
            );
            IF v_most_imbalanced_region IS NOT NULL THEN
                SET v_region_sku_count = (
                    SELECT COUNT(*)
                    FROM products p
                    JOIN suppliers s ON p.supplier_id = s.supplier_id
                    WHERE s.supplier_region = v_most_imbalanced_region
                );
                WHILE v_region_sku_count < 4 DO
                    INSERT INTO suppliers VALUES (
                        (SELECT MAX(supplier_id) + 1 FROM suppliers),
                        'Balancer ' || v_most_imbalanced_region,
                        'Balancer Contact',
                        'Balancer Role',
                        'balancer@regional.com',
                        '555-0000',
                        '123 Balance St',
                        'Balancer City',
                        'CA',
                        '00000',
                        'USA',
                        v_most_imbalanced_region,
                        'Balancer',
                        'http://balancer.com',
                        CURRENT_TIMESTAMP,
                        NULL,
                        NULL,
                        4.5,
                        true
                    );
                    INSERT INTO products VALUES (
                        (SELECT MAX(product_id) + 1 FROM products),
                        'Region SKU',
                        'Balancer',
                        29.99,
                        19.99,
                        (SELECT MAX(supplier_id) FROM suppliers),
                        'Balancer Contact',
                        'Balancer Role',
                        '123 Balance St',
                        4,
                        'Black',
                        'Medium',
                        0.6,
                        'Region balancer',
                        3.0,
                        0,
                        CURRENT_TIMESTAMP,
                        CURRENT_TIMESTAMP,
                        true,
                        0.0,
                        0.19
                    );
                    SET v_region_sku_count = v_region_sku_count + 1;
                END WHILE;
            END IF;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: IF, WHILE, DML, deep-nesting, exception-handler, region-sku-balance-logic
-- EXPECTED: Regional SKU balancing, supplier/product inserts, DML
-- EXECUTES: IF, WHILE, nested BEGIN...END, exception handler

-- Script 81
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_item_bulk_flag BOOLEAN DEFAULT false;
    DECLARE v_large_orders_count INT;
    DECLARE v_min_bulk_items INT DEFAULT 2;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Bulk item error'); END;

    L0: BEGIN
        SET v_large_orders_count = (SELECT COUNT(*) FROM order_items WHERE item_quantity >= 5);
        IF v_large_orders_count < v_min_bulk_items THEN
            WHILE v_large_orders_count < v_min_bulk_items DO
                INSERT INTO orders (
                    order_id,
                    customer_id,
                    order_date,
                    order_status,
                    order_total,
                    order_tax,
                    order_discount,
                    order_shipping
                ) VALUES (
                    (SELECT MAX(order_id) + 1 FROM orders),
                    (SELECT MIN(customer_id) FROM customers),
                    CURRENT_TIMESTAMP,
                    'Bulk',
                    120.0,
                    12.0,
                    0.0,
                    0.0
                );
                INSERT INTO order_items (
                    order_item_id,
                    order_id,
                    product_id,
                    item_quantity,
                    item_price_per_unit,
                    item_discount,
                    item_tax,
                    item_total
                ) VALUES (
                    (SELECT MAX(order_item_id) + 1 FROM order_items),
                    (SELECT MAX(order_id) FROM orders),
                    (SELECT MIN(product_id) FROM products),
                    6,
                    15.0,
                    0.0,
                    1.0,
                    96.0
                );
                SET v_large_orders_count = v_large_orders_count + 1;
            END WHILE;
            SET v_order_item_bulk_flag = true;
        END IF;
        IF v_order_item_bulk_flag = false AND (SELECT COUNT(*) FROM order_items WHERE item_quantity > 1) < 5 THEN
            REPEAT
                INSERT INTO order_items (
                    order_item_id,
                    order_id,
                    product_id,
                    item_quantity,
                    item_price_per_unit,
                    item_discount,
                    item_tax,
                    item_total
                ) VALUES (
                    (SELECT MAX(order_item_id) + 1 FROM order_items),
                    (SELECT MIN(order_id) FROM orders),
                    (SELECT MIN(product_id) FROM products),
                    3,
                    12.0,
                    0.0,
                    1.0,
                    39.0
                );
            UNTIL (SELECT COUNT(*) FROM order_items WHERE item_quantity > 1) >= 5
            END REPEAT;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: WHILE, REPEAT, IF, DML, deep-nesting, exception-handler, bulk-item-coverage
-- EXPECTED: Order/item inserts for bulk sales, DML
-- EXECUTES: WHILE, REPEAT, IF, nested BEGIN...END, exception handler

-- Script 82
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_company_flag_count INT;
    DECLARE V_COMPANY_FLAG_KEY STRING DEFAULT 'is_company';
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Company customer error'); END;

    L0: BEGIN
        SET v_customer_company_flag_count = (SELECT COUNT(*) FROM customers WHERE JSON_EXTRACT(customer_metadata, V_COMPANY_FLAG_KEY) = 'true');
        IF v_customer_company_flag_count < 2 THEN
            L1: BEGIN
                FOR cust AS SELECT customer_id FROM customers WHERE JSON_EXTRACT(customer_metadata, V_COMPANY_FLAG_KEY) IS NULL OR JSON_EXTRACT(customer_metadata, V_COMPANY_FLAG_KEY) = 'false' LIMIT 2 DO
                    UPDATE customers
                    SET customer_metadata = JSON_SET(customer_metadata, '$.' || V_COMPANY_FLAG_KEY, 'true')
                    WHERE customer_id = cust.customer_id;
                    SET v_customer_company_flag_count = v_customer_company_flag_count + 1;
                END FOR;
                IF v_customer_company_flag_count < 1 THEN
                    INSERT INTO customers (customer_id, customer_name, customer_metadata)
                    VALUES (
                        (SELECT MAX(customer_id) + 1 FROM customers),
                        'TopLevel Inc.',
                        JSON_OBJECT(V_COMPANY_FLAG_KEY, 'true')
                    );
                END IF;
            END L1;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, JSON, DML, deep-nesting, exception-handler, company-flag-logic
-- EXPECTED: Customer metadata updates, company flag logic, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 83
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_item_comment_required BOOLEAN DEFAULT false;
    DECLARE v_order_item_with_comments INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Order item comment error'); END;

    L0: BEGIN
        SET v_order_item_with_comments = (SELECT COUNT(*) FROM order_items WHERE item_notes IS NOT NULL AND item_notes != '');
        IF v_order_item_with_comments < 3 THEN
            FOR oi AS SELECT order_item_id FROM order_items WHERE item_notes IS NULL OR item_notes = '' LIMIT 3 DO
                UPDATE order_items
                SET item_notes = CASE
                                    WHEN MOD(oi.order_item_id, 2) = 0 THEN 'Custom request'
                                    WHEN MOD(oi.order_item_id, 3) = 0 THEN 'Special handling'
                                    ELSE 'No comment'
                                 END
                WHERE order_item_id = oi.order_item_id;
                SET v_order_item_with_comments = v_order_item_with_comments + 1;
                SET v_order_item_comment_required = true;
            END FOR;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, item-comment-logic
-- EXPECTED: Order item note filling, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 84
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_tags_count INT;
    DECLARE V_TAG_VALUE STRING DEFAULT 'featured';
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Product tag error'); END;

    L0: BEGIN
        SET v_product_tags_count = (SELECT COUNT(*) FROM products WHERE JSON_EXTRACT(product_metadata, '$.tags') IS NOT NULL AND JSON_EXTRACT(product_metadata, '$.tags') != '[]');
        IF v_product_tags_count < 3 THEN
            L1: BEGIN
                FOR prod AS SELECT product_id, product_metadata FROM products LIMIT 3 DO
                    IF JSON_EXTRACT(prod.product_metadata, '$.tags') IS NULL OR JSON_EXTRACT(prod.product_metadata, '$.tags') = '[]' THEN
                        UPDATE products
                        SET product_metadata = JSON_SET(
                            IFNULL(product_metadata, '{}'),
                            '$.tags',
                            JSON_ARRAY(
                                V_TAG_VALUE,
                                CASE WHEN MOD(prod.product_id, 2) = 0 THEN 'eco' ELSE 'limited' END
                            )
                        )
                        WHERE product_id = prod.product_id;
                        SET v_product_tags_count = v_product_tags_count + 1;
                    END IF;
                END FOR;
            END L1;
        END IF;
        IF v_product_tags_count < 1 THEN
            INSERT INTO products VALUES (
                (SELECT MAX(product_id) + 1 FROM products),
                'Tagged Example',
                'Special',
                29.99,
                19.99,
                (SELECT MIN(supplier_id) FROM suppliers),
                'Tag Contact',
                'Tag Role',
                '123 Tag St',
                5,
                'Yellow',
                'Small',
                0.5,
                'Tagged',
                4.0,
                0,
                CURRENT_TIMESTAMP,
                CURRENT_TIMESTAMP,
                true,
                0.0,
                0.19,
                '{"tags": ["featured", "eco"]}'
            );
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, JSON, DML, deep-nesting, exception-handler, product-tag-logic
-- EXPECTED: Product metadata tag updates, tagged product inserts, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 85
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_multiple_statuses BOOLEAN DEFAULT false;
    DECLARE v_different_workflows INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Order workflow error'); END;

    L0: BEGIN
        SET v_different_workflows = (SELECT COUNT(DISTINCT order_workflow_type) FROM orders WHERE order_workflow_type IS NOT NULL);
        IF v_different_workflows < 2 THEN
            L1: BEGIN
                FOR ord AS SELECT order_id FROM orders WHERE order_workflow_type IS NULL LIMIT 2 DO
                    UPDATE orders
                    SET order_workflow_type = CASE
                                                WHEN MOD(ord.order_id, 2) = 0 THEN 'fast_track'
                                                ELSE 'standard'
                                             END
                    WHERE order_id = ord.order_id;
                END FOR;
                SET v_order_multiple_statuses = true;
            END L1;
        END IF;
        IF v_order_multiple_statuses = false THEN
            INSERT INTO orders (
                order_id,
                customer_id,
                order_date,
                order_status,
                order_total,
                order_tax,
                order_discount,
                order_shipping,
                order_workflow_type
            ) VALUES (
                (SELECT MAX(order_id) + 1 FROM orders),
                (SELECT MIN(customer_id) FROM customers),
                CURRENT_TIMESTAMP,
                'Custom Workflow',
                12.0,
                1.2,
                0.0,
                2.0,
                'bulk_processing'
            );
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, workflow-diversity
-- EXPECTED: Order workflow type diversity, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 86
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_inactive_supplier_count INT;
    DECLARE v_total_products_from_inactive_suppliers INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Supplier inactivity error'); END;

    L0: BEGIN
        SET v_inactive_supplier_count = (SELECT COUNT(*) FROM suppliers WHERE supplier_active = false);
        SET v_total_products_from_inactive_suppliers = (SELECT COUNT(*) FROM products p JOIN suppliers s ON p.supplier_id = s.supplier_id WHERE s.supplier_active = false);
        IF v_inactive_supplier_count > 1 AND v_total_products_from_inactive_suppliers > 1 THEN
            L1: BEGIN
                DECLARE v_inactive_supplier_id INT;
                SET v_inactive_supprier_id = (SELECT MIN(supplier_id) FROM suppliers WHERE supplier_active = false LIMIT 1);
                INSERT INTO products VALUES (
                    (SELECT MAX(product_id) + 1 FROM products),
                    'Clearance',
                    'Closeout',
                    9.99,
                    4.99,
                    v_inactive_supplier_id,
                    'Clearance Contact',
                    'Clearance Role',
                    '123 Clearance St',
                    1,
                    'Gray',
                    'Small',
                    0.1,
                    'Supplier inactive',
                    0.0,
                    0,
                    CURRENT_TIMESTAMP,
                    CURRENT_TIMESTAMP,
                    true,
                    0.5,
                    0.19
                );
                SET v_inactive_supplier_count = v_inactive_supplier_count - 1;
                SET v_total_products_from_inactive_suppliers = v_total_products_from_inactive_suppliers + 1;
            END L1;
        ELSE
            L2: BEGIN
                INSERT INTO suppliers VALUES (
                    (SELECT MAX(supplier_id) + 1 FROM suppliers),
                    'Inactive Test',
                    'Inactive Manager',
                    'Inactive Role',
                    'inactive@test.com',
                    '555-0000',
                    '123 Inactive St',
                    'Inactive City',
                    'CA',
                    '00000',
                    'USA',
                    'Test',
                    'General',
                    NULL,
                    CURRENT_TIMESTAMP,
                    NULL,
                    NULL,
                    2.5,
                    false
                );
                SET v_inactive_supplier_count = v_inactive_supplier_count + 1;
            END L2;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: IF, DML, deep-nesting, exception-handler, inactive-supplier-logic
-- EXPECTED: Clearance product inserts, inactive supplier coverage, DML
-- EXECUTES: IF, nested BEGIN...END, exception handler

-- Script 87
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_item_returns BOOLEAN DEFAULT false;
    DECLARE v_returns_to_cover INT;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Return item error'); END;

    L0: BEGIN
        SET v_returns_to_cover = (SELECT COUNT(*) FROM order_items WHERE item_status = 'Returned');
        IF v_returns_to_cover < 2 THEN
            L1: BEGIN
                FOR oi AS SELECT order_item_id FROM order_items WHERE item_status != 'Returned' LIMIT 2 DO
                    UPDATE order_items
                    SET item_status = 'Returned',
                        item_return_date = CURRENT_TIMESTAMP
                    WHERE order_item_id = oi.order_item_id;
                    SET v_returns_to_cover = v_returns_to_cover + 1;
                END FOR;
                SET v_order_item_returns = true;
            END L1;
        END IF;
        IF v_order_item_returns = false AND v_returns_to_cover < 1 THEN
            INSERT INTO order_items (
                order_item_id,
                order_id,
                product_id,
                item_quantity,
                item_price_per_unit,
                item_discount,
                item_tax,
                item_total,
                item_status,
                item_return_date
            ) VALUES (
                (SELECT MAX(order_item_id) + 1 FROM order_items),
                (SELECT MIN(order_id) FROM orders),
                (SELECT MIN(product_id) FROM products),
                1,
                10.0,
                0.0,
                1.0,
                11.0,
                'Returned',
                CURRENT_TIMESTAMP
            );
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, return-logic
-- EXPECTED: Returned item updates, return state coverage, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 88
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_orders_no_discount INT;
    DECLARE v_discount_provided BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Discount logic error'); END;

    L0: BEGIN
        SET v_customer_orders_no_discount = (
            SELECT COUNT(*)
            FROM customers c
            JOIN orders o ON c.customer_id = o.customer_id
            WHERE o.order_discount = 0
        );
        IF v_customer_orders_no_discount > 2 THEN
            L1: BEGIN
                FOR cust AS
                    SELECT c.customer_id
                    FROM customers c
                    JOIN orders o ON c.customer_id = o.customer_id
                    WHERE o.order_discount = 0
                    GROUP BY c.customer_id
                    LIMIT 2
                DO
                    UPDATE orders
                    SET order_discount = 0.1,
                        order_status = 'Discount Applied'
                    WHERE customer_id = cust.customer_id
                    AND order_discount = 0;
                    SET v_discount_provided = true;
                END FOR;
            END L1;
        END IF;
        IF v_discount_provided = false AND v_customer_orders_no_discount > 0 THEN
            INSERT INTO orders (
                order_id,
                customer_id,
                order_date,
                order_status,
                order_total,
                order_tax,
                order_discount,
                order_shipping
            ) VALUES (
                (SELECT MAX(order_id) + 1 FROM orders),
                (SELECT MIN(customer_id) FROM customers WHERE NOT EXISTS (SELECT 1 FROM orders WHERE customer_id = customers.customer_id)),
                CURRENT_TIMESTAMP,
                'Discount Provided',
                25.0,
                2.5,
                0.15,
                2.5
            );
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, discount-coverage
-- EXPECTED: Discount application, discount coverage logic, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 89
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_supplier_product_threshold INT DEFAULT 3;
    DECLARE v_critical_category_products INT;
    DECLARE v_extra_promos_flag BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Critical category error'); END;

    L0: BEGIN
        FOR category AS SELECT DISTINCT product_category FROM products WHERE product_category != 'General' LIMIT 2 DO
            SET v_critical_category_products = (SELECT COUNT(*) FROM products WHERE product_category = category.product_category);
            IF v_critical_category_products < v_supplier_product_threshold THEN
                -- Insert additional products to reach threshold
                WHILE v_critical_category_products < v_supplier_product_threshold DO
                    INSERT INTO products VALUES (
                        (SELECT MAX(product_id) + 1 FROM products),
                        'Critical ' || category.product_category,
                        category.product_category,
                        34.99,
                        24.99,
                        (SELECT MIN(supplier_id) FROM suppliers),
                        'Critical Contact',
                        'Critical Role',
                        '123 Critical St',
                        7,
                        'Silver',
                        'Medium',
                        0.8,
                        'Critical',
                        4.5,
                        0,
                        CURRENT_TIMESTAMP,
                        CURRENT_TIMESTAMP,
                        true,
                        0.0,
                        0.19
                    );
                    SET v_critical_category_products = (SELECT COUNT(*) FROM products WHERE product_category = category.product_category);
                END WHILE;
                SET v_extra_promos_flag = true;
            END IF;
        END FOR;
        IF v_extra_promos_flag = true THEN
            -- To ensure orders for these categories
            FOR category AS SELECT DISTINCT product_category FROM products LIMIT 1 DO
                IF NOT EXISTS (SELECT 1 FROM order_items oi JOIN products p ON oi.product_id = p.product_id WHERE p.product_category = category.product_category) THEN
                    INSERT INTO order_items (
                        order_item_id,
                        order_id,
                        product_id,
                        item_quantity,
                        item_price_per_unit,
                        item_discount,
                        item_tax,
                        item_total
                    ) VALUES (
                        (SELECT MAX(order_item_id) + 1 FROM order_items),
                        (SELECT MIN(order_id) FROM orders),
                        (SELECT MIN(product_id) FROM products WHERE product_category = category.product_category),
                        2,
                        30.0,
                        0.0,
                        3.0,
                        66.0
                    );
                END IF;
            END FOR;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, WHILE, DML, deep-nesting, exception-handler, critical-category-logic
-- EXPECTED: Critical category product inserts, order item coverage, DML
-- EXECUTES: FOR, IF, WHILE, nested BEGIN...END, exception handler

-- Script 90
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_item_status_aging BOOLEAN DEFAULT false;
    DECLARE v_order_item_status_preprocessing INT;
    DECLARE v_required_status_transitions INT DEFAULT 2;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Status workflow error'); END;

    L0: BEGIN
        SET v_order_item_status_preprocessing = (SELECT COUNT(*) FROM order_items WHERE item_status = 'Preprocessing' AND item_delivery_date IS NULL);
        IF v_order_item_status_preprocessing >= 1 THEN
            -- Advance one preprocessing item
            UPDATE order_items
            SET item_status = 'Processing',
                item_delivery_date = CURRENT_TIMESTAMP
            WHERE order_item_id = (SELECT MIN(order_item_id) FROM order_items WHERE item_status = 'Preprocessing');
            SET v_order_item_status_aging = true;
        END IF;
        IF v_order_item_status_aging = false AND (SELECT COUNT(*) FROM customers WHERE customer_last_order_date > DATEADD(DAY, -7, CURRENT_TIMESTAMP)) < 2 THEN
            INSERT INTO orders (
                order_id,
                customer_id,
                order_date,
                order_status,
                order_total,
                order_tax,
                order_discount,
                order_shipping
            ) VALUES (
                (SELECT MAX(order_id) + 1 FROM orders),
                (SELECT MIN(customer_id) FROM customers),
                CURRENT_TIMESTAMP,
                'Item Status Flow',
                15.0,
                1.5,
                0.0,
                2.0
            );
            INSERT INTO order_items (
                order_item_id,
                order_id,
                product_id,
                item_quantity,
                item_price_per_unit,
                item_discount,
                item_tax,
                item_total,
                item_status,
                item_delivery_date
            ) VALUES (
                (SELECT MAX(order_item_id) + 1 FROM order_items),
                (SELECT MAX(order_id) FROM orders),
                (SELECT MIN(product_id) FROM products),
                1,
                12.0,
                0.0,
                1.0,
                13.0,
                'Preprocessing',
                NULL
            );
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: IF, DML, deep-nesting, exception-handler, status-workflow-logic
-- EXPECTED: Status workflow updates, order and item inserts, DML
-- EXECUTES: IF, nested BEGIN...END, exception handler

-- Script 91
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_product_missing_weight INT;
    DECLARE v_product_weight_coverage BOOLEAN DEFAULT false;
    DECLARE v_min_weight_products INT DEFAULT 2;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Product weight error'); END;

    L0: BEGIN
        SET v_product_missing_weight = (SELECT COUNT(*) FROM products WHERE product_weight IS NULL AND product_category NOT LIKE '%Accessories%');
        IF v_product_missing_weight > 0 THEN
            L1: BEGIN
                FOR prod AS SELECT product_id FROM products WHERE product_weight IS NULL AND product_category NOT LIKE '%Accessories%' LIMIT v_min_weight_products DO
                    UPDATE products
                    SET product_weight = ROUND(RAND() * 2 + 0.2, 2)
                    WHERE product_id = prod.product_id;
                    SET v_product_weight_coverage = true;
                END FOR;
            END L1;
        END IF;
        IF v_product_weight_coverage = false AND v_product_missing_weight > 0 THEN
            INSERT INTO products VALUES (
                (SELECT MAX(product_id) + 1 FROM products),
                'Weighted Product',
                'General',
                29.99,
                19.99,
                (SELECT MIN(supplier_id) FROM suppliers),
                'Weight Contact',
                'Weight Role',
                '123 Weight St',
                6,
                'Black',
                'Medium',
                2.2,
                'Weighted sample',
                3.5,
                0,
                CURRENT_TIMESTAMP,
                CURRENT_TIMESTAMP,
                true,
                0.0,
                0.19
            );
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, product-weight-logic
-- EXPECTED: Product weight updates, weighted product inserts, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 92
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_opt_in_newsletter_count INT;
    DECLARE v_opt_in_diversity BOOLEAN DEFAULT false;
    DECLARE v_min_opt_in_variety INT DEFAULT 2;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Newsletter opt-in error'); END;

    L0: BEGIN
        SET v_customer_opt_in_newsletter_count = (SELECT COUNT(*) FROM customers WHERE customer_opt_in_newsletter = true);
        IF v_customer_opt_in_newsletter_count < v_min_opt_in_variety THEN
            L1: BEGIN
                FOR cust AS SELECT customer_id FROM customers WHERE customer_opt_in_newsletter IS NULL LIMIT v_min_opt_in_variety DO
                    UPDATE customers
                    SET customer_opt_in_newsletter = true
                    WHERE customer_id = cust.customer_id;
                    SET v_opt_in_diversity = true;
                END FOR;
                IF v_opt_in_diversity = false AND v_customer_opt_in_newsletter_count < v_min_opt_in_variety THEN
                    INSERT INTO customers VALUES (
                        (SELECT MAX(customer_id) + 1 FROM customers),
                        'Opt-In Customer',
                        'optin@example.com',
                        '555-0000',
                        '123 OptIn St',
                        'OptIn City',
                        'CA',
                        '00000',
                        'USA',
                        'Retail',
                        CURRENT_TIMESTAMP,
                        0.0,
                        NULL,
                        NULL,
                        NULL,
                        true,
                        true,
                        0.0,
                        NULL,
                        NULL
                    );
                END IF;
            END L1;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, newsletter-opt-in-logic
-- EXPECTED: Newsletter opt-in diversity, customer updates/inserts, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 93
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_supplier_product_diversity_count INT;
    DECLARE v_min_supplier_product_diversity INT DEFAULT 2;
    DECLARE v_diversity_enforced BOOLEAN DEFAULT false;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Supplier-product diversity error'); END;

    L0: BEGIN
        SET v_supplier_product_diversity_count = (
            SELECT COUNT(DISTINCT product_category)
            FROM products p
            JOIN suppliers s ON p.supplier_id = s.supplier_id
            WHERE s.supplier_main_products IS NULL
               OR p.product_category NOT LIKE CONCAT('%', s.supplier_main_products, '%')
        );
        IF v_supplier_product_diversity_count < v_min_supplier_product_diversity THEN
            FOR sup AS
                SELECT s.supplier_id, s.supplier_main_products
                FROM suppliers s
                WHERE s.supplier_main_products IS NOT NULL
                  AND s.supplier_main_products != 'General'
                LIMIT 1
            DO
                INSERT INTO products VALUES (
                    (SELECT MAX(product_id) + 1 FROM products),
                    'Diverse Category Product',
                    (SELECT MIN(product_category) FROM products
                     WHERE product_category NOT LIKE CONCAT('%', sup.supplier_main_products, '%')
                     AND product_category IS NOT NULL),
                    29.99,
                    19.99,
                    sup.supplier_id,
                    'Diverse Contact',
                    'Diverse Role',
                    '123 Diverse St',
                    4,
                    'White',
                    'Medium',
                    0.7,
                    'Supplier-product diversity',
                    3.0,
                    0,
                    CURRENT_TIMESTAMP,
                    CURRENT_TIMESTAMP,
                    true,
                    0.0,
                    0.19
                );
            END FOR;
            SET v_diversity_enforced = true;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOR, IF, DML, deep-nesting, exception-handler, supplier-product-diversity-logic
-- EXPECTED: Supplier-product category diversity inserts, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

-- Script 94
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_order_item_bundled_flag BOOLEAN DEFAULT false;
    DECLARE v_bundled_item_types_count INT;
    DECLARE v_min_bundled_items INT DEFAULT 1;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Bundle error'); END;

    L0: BEGIN
        SET v_bundled_item_types_count = (SELECT COUNT(*) FROM products WHERE product_category = 'Bundle');
        IF v_bundled_item_types_count < v_min_bundled_items THEN
            INSERT INTO products VALUES (
                (SELECT MAX(product_id) + 1 FROM products),
                'Bundle Starter',
                'Bundle',
                49.99,
                39.99,
                (SELECT MIN(supplier_id) FROM suppliers),
                'Bundle Contact',
                'Bundle Role',
                '123 Bundle St',
                5,
                'Multi',
                'Set',
                2.0,
                'Bundle to bundle',
                4.0,
                0,
                CURRENT_TIMESTAMP,
                CURRENT_TIMESTAMP,
                true,
                0.0,
                0.19
            );
            SET v_bundled_item_types_count = v_bundled_item_types_count + 1;
            SET v_order_item_bundled_flag = true;
        END IF;
        IF v_order_item_bundled_flag = false THEN
            REPEAT
                INSERT INTO order_items (
                    order_item_id,
                    order_id,
                    product_id,
                    item_quantity,
                    item_price_per_unit,
                    item_discount,
                    item_tax,
                    item_total,
                    item_bundle_notes
                ) VALUES (
                    (SELECT MAX(order_item_id) + 1 FROM order_items),
                    (SELECT MIN(order_id) FROM orders),
                    (SELECT MAX(product_id) FROM products WHERE product_category = 'Bundle'),
                    1,
                    49.99,
                    0.0,
                    5.0,
                    54.99,
                    'From bundle logic'
                );
            UNTIL (SELECT COUNT(*) FROM order_items WHERE item_bundle_notes IS NOT NULL) >= v_min_bundled_items
            END REPEAT;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: IF, REPEAT, DML, deep-nesting, exception-handler, bundled-item-logic
-- EXPECTED: Product bundle inserts, bundle order item inserts, DML
-- EXECUTES: IF, REPEAT, nested BEGIN...END, exception handler

-- Script 95
--QUERY-DELIMITER-START
BEGIN
    DECLARE v_customer_is_foreign_flag BOOLEAN DEFAULT false;
    DECLARE v_foreign_customer_count INT;
    DECLARE v_min_foreign_customers INT DEFAULT 2;
    DECLARE EXIT HANDLER FOR SQLEXCEPTION BEGIN VALUES('Foreign customer error'); END;

    L0: BEGIN
        SET v_foreign_customer_count = (SELECT COUNT(*) FROM customers WHERE customer_country != 'USA');
        IF v_foreign_customer_count < v_min_foreign_customers THEN
            FOR country AS SELECT DISTINCT supplier_country FROM suppliers WHERE supplier_country != 'USA' LIMIT 2 DO
                INSERT INTO customers (
                    customer_id,
                    customer_name,
                    customer_email,
                    customer_phone,
                    customer_address_1,
                    customer_city,
                    customer_state,
                    customer_zip,
                    customer_country,
                    customer_type,
                    customer_total_spent,
                    customer_active_status,
                    customer_opt_in_newsletter
                ) VALUES (
                    (SELECT MAX(customer_id) + 1 FROM customers),
                    'International ' || country.supplier_country,
                    'intl@' || LOWER(country.supplier_country) || '.com',
                    '555-0000',
                    '123 Intl St',
                    'International City',
                    country.supplier_country,
                    '00000',
                    country.supplier_country,
                    'Retail',
                    0.0,
                    true,
                    true
                );
                SET v_customer_is_foreign_flag = true;
            END FOR;
        END IF;
    END L0;
END;
--QUERY-DELIMITER-END
-- TAGS: FOREACH, IF, DML, deep-nesting, exception-handler, foreign-customer-logic
-- EXPECTED: Foreign customer inserts, international customer coverage, DML
-- EXECUTES: FOR, IF, nested BEGIN...END, exception handler

DROP TABLE products;
DROP TABLE customers;
DROP TABLE orders;
DROP TABLE order_items;
DROP TABLE employees;
DROP TABLE suppliers;