// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.trees.expressions.functions.scalar;

import org.apache.doris.catalog.FunctionSignature;
import org.apache.doris.nereids.trees.expressions.Expression;
import org.apache.doris.nereids.trees.expressions.functions.ExplicitlyCastableSignature;
import org.apache.doris.nereids.trees.expressions.functions.Monotonic;
import org.apache.doris.nereids.trees.expressions.functions.PropagateNullLiteral;
import org.apache.doris.nereids.trees.expressions.functions.PropagateNullable;
import org.apache.doris.nereids.trees.expressions.literal.Literal;
import org.apache.doris.nereids.trees.expressions.literal.StringLikeLiteral;
import org.apache.doris.nereids.trees.expressions.visitor.ExpressionVisitor;
import org.apache.doris.nereids.types.BigIntType;
import org.apache.doris.nereids.types.DecimalV3Type;
import org.apache.doris.nereids.types.StringType;
import org.apache.doris.nereids.types.VarcharType;
import org.apache.doris.nereids.util.DateUtils;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;

import java.util.List;

/**
 * ScalarFunction 'from_unixtime'. This class is generated by GenerateFunction.
 */
public class FromUnixtime extends ScalarFunction
        implements ExplicitlyCastableSignature, PropagateNullable, PropagateNullLiteral, Monotonic {
    public static final List<FunctionSignature> SIGNATURES = ImmutableList.of(
            FunctionSignature.ret(VarcharType.SYSTEM_DEFAULT).args(BigIntType.INSTANCE),
            FunctionSignature.ret(VarcharType.SYSTEM_DEFAULT).args(BigIntType.INSTANCE, VarcharType.SYSTEM_DEFAULT),
            FunctionSignature.ret(VarcharType.SYSTEM_DEFAULT).args(BigIntType.INSTANCE, StringType.INSTANCE),
            FunctionSignature.ret(VarcharType.SYSTEM_DEFAULT).args(DecimalV3Type.createDecimalV3Type(18, 6)),
            FunctionSignature.ret(VarcharType.SYSTEM_DEFAULT).args(DecimalV3Type.createDecimalV3Type(18, 6),
                    VarcharType.SYSTEM_DEFAULT),
            FunctionSignature.ret(VarcharType.SYSTEM_DEFAULT).args(DecimalV3Type.createDecimalV3Type(18, 6),
                    StringType.INSTANCE));

    /**
     * constructor with 1 argument.
     */
    public FromUnixtime(Expression arg) {
        super("from_unixtime", arg);
    }

    /**
     * constructor with 2 arguments.
     */
    public FromUnixtime(Expression arg0, Expression arg1) {
        super("from_unixtime", arg0, arg1);
    }

    /** constructor for withChildren and reuse signature */
    private FromUnixtime(ScalarFunctionParams functionParams) {
        super(functionParams);
    }

    /**
     * withChildren.
     */
    @Override
    public FromUnixtime withChildren(List<Expression> children) {
        Preconditions.checkArgument(children.size() == 1 || children.size() == 2);
        return new FromUnixtime(getFunctionParams(children));
    }

    @Override
    public List<FunctionSignature> getSignatures() {
        return SIGNATURES;
    }

    @Override
    public FunctionSignature computeSignature(FunctionSignature signature) {
        // skip super.computeSignature() to avoid changing the decimal precision
        // manually set decimal argument's type to always decimal(18, 6)
        if (this.getArgumentType(0).isDecimalLikeType()) {
            Preconditions.checkArgument(arity() == 1 || arity() == 2, "FromUnixtime should have 1 or 2 arguments");
            if (arity() == 1) {
                return FunctionSignature.ret(VarcharType.SYSTEM_DEFAULT)
                        .args(DecimalV3Type.createDecimalV3Type(18, 6));
            } else {
                return FunctionSignature.ret(VarcharType.SYSTEM_DEFAULT)
                        .args(DecimalV3Type.createDecimalV3Type(18, 6), VarcharType.SYSTEM_DEFAULT);
            }
        }
        return signature;
    }

    @Override
    public <R, C> R accept(ExpressionVisitor<R, C> visitor, C context) {
        return visitor.visitFromUnixtime(this, context);
    }

    @Override
    public boolean isMonotonic(Literal lower, Literal upper) {
        if (1 == arity()) {
            return true;
        }
        Expression format = child(1);
        if (!(format instanceof StringLikeLiteral)) {
            return false;
        }
        String str = ((StringLikeLiteral) format).getValue();
        return DateUtils.monoFormat.contains(str);
    }

    @Override
    public boolean isPositive() {
        return true;
    }

    @Override
    public int getMonotonicFunctionChildIndex() {
        return 0;
    }

    @Override
    public Expression withConstantArgs(Expression literal) {
        if (1 == arity()) {
            return new FromUnixtime(literal);
        }
        return new FromUnixtime(literal, child(1));
    }
}
